/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

package com.salelder;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.openimaj.video.Video;
import org.openimaj.video.capture.Device;
import org.openimaj.video.capture.VideoCapture;

public class OptionsPanel extends JPanel {
	private static int FIELD_WIDTH = 10;
	//private JTextField title;
	private JButton loadButton, viewButton, edgeButton;
	private JComboBox<String> webcamSelector;
	private JSlider opacitySlider, brightnessSlider, contrastSlider, scaleSlider, panXSlider, panYSlider, tempSlider;
	private JSpinner camWidth, camHeight;
	private PreviewPanel preview;
	private PreviewController previewController;
	private LoadImageListener loadListener;
	private List<Device> camList;
	private JButton refresh;
	
	public JFileChooser fileChooser;
	private License license;
	
	public OptionsPanel(PreviewPanel p, License l) {
		setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
		preview = p;
		license= l;
		previewController = new PreviewController(preview, this);;
		
		fileChooser = new JFileChooser();		
		loadButton = new JButton("Add reference image");
		loadListener = new LoadImageListener(preview, this, license);
		loadButton.addActionListener(loadListener);
		
		viewButton = new JButton("Toggle view");
		viewButton.addActionListener(previewController);
		
		edgeButton = new JButton("Edge detection on/off");
		edgeButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent arg0) {
				preview.toggleEdges();				
			}});

		webcamSelector = new JComboBox<String>();
		updateWebcams();
		webcamSelector.addItemListener(previewController);
		
		opacitySlider = new JSlider(0, 100);
		opacitySlider.addChangeListener(previewController);
		
		brightnessSlider = new JSlider(0, 300, 100);
		brightnessSlider.addChangeListener(previewController);
		contrastSlider = new JSlider(0, 300, 100);
		contrastSlider.addChangeListener(previewController);
		tempSlider = new JSlider(2000, 20000, 6500);
		tempSlider.addChangeListener(previewController);
		
		scaleSlider = new JSlider(50, 300, 100);
		scaleSlider.addChangeListener(previewController);	
		panXSlider = new JSlider(-75, 75, 0);
		panXSlider.addChangeListener(previewController);	
		panYSlider = new JSlider(-75, 75, 0);
		panYSlider.addChangeListener(previewController);
		
		refresh = new JButton("Refresh webcam list");
		refresh.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				updateWebcams();
			}
		});
		
		camWidth = new JSpinner(new SpinnerNumberModel(PreviewPanel.VIDEO_WIDTH, 200, 2000, 10));
		camHeight = new JSpinner(new SpinnerNumberModel(PreviewPanel.VIDEO_HEIGHT, 200, 2000, 10));
		ChangeListener camSizeListener = new ChangeListener() {
			@Override
			public void stateChanged(ChangeEvent arg0) {
				preview.setWebcam(getWebcam(), (Integer)camWidth.getValue(), (Integer)camHeight.getValue());
				
			}};
		camWidth.addChangeListener(camSizeListener);
		camHeight.addChangeListener(camSizeListener);
		
		//add(title);
		add(loadButton);
		add(viewButton);
		add(edgeButton);
		add(new JLabel("Opacity")); add(opacitySlider);
		add(webcamSelector);
		add(refresh);
		add(new JLabel("Brightness")); add(brightnessSlider);
		add(new JLabel("Contrast")); add(contrastSlider);
		add(new JLabel("Color temperature")); add(tempSlider);
		add(new JLabel("Scale")); add(scaleSlider);
		add(new JLabel("Pan X")); add(panXSlider);
		add(new JLabel("Pan Y")); add(panYSlider);
		add(new JLabel("Webcam dimensions: "));
		add(camWidth); add(camHeight);
	}
	public PreviewController getPreviewController() {
		return previewController;
	}
	
	public void toggleOpacity() {
		previewController.toggleOpacity();
		opacitySlider.setValue((int)(100*preview.getOpacity()));
	}
	
	public float getOpacity() {
		return 0.01f * opacitySlider.getValue();
	}
	public float getBrightness() {
		return 0.01f * brightnessSlider.getValue();
	}
	public float getContrast() {
		return .01f * contrastSlider.getValue();
	}
	public float getScale() {
		return .01f * scaleSlider.getValue();
	}
	public float getPanX() {
		return .01f * panXSlider.getValue();
	}
	public float getPanY() {
		return .01f * panYSlider.getValue();
	}
	public float getColorTemp() {
		// TODO Auto-generated method stub
		return tempSlider.getValue();
	}
	public int getVideoWidth() {
		return (Integer) camWidth.getValue();
	}
	public int getVideoHeight() {
		return (Integer) camHeight.getValue();
	}
	
	public void setParams(float brightness, float contrast, float scale, float panX, float panY, double temp, int videoWidth, int videoHeight) {
		/* Update the sliders and trigger a change in the preview panel. */
		brightnessSlider.setValue((int) (brightness * 100));
		contrastSlider.setValue((int) (contrast*100));
		scaleSlider.setValue((int)(scale*100));
		panXSlider.setValue((int)(panX*100));
		panYSlider.setValue((int)(panY*100));
		tempSlider.setValue((int)temp);
		previewController.stateChanged(null); // trigger update
		
		camWidth.setValue(videoWidth);
		camHeight.setValue(videoHeight);
		previewController.itemStateChanged(null);
	}
	public void updateWebcams() {
	
		/* Check available devices. */
		camList = VideoCapture.getVideoDevices();
		webcamSelector.removeAllItems();
		for (Device d : camList) {
			webcamSelector.addItem(d.getIdentifierStr());
		}
		
	}
	public Device getWebcam() {
		int k = webcamSelector.getSelectedIndex();
		return camList.get(k);
	}
}
