/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

package com.salelder;

import org.openimaj.image.pixel.Pixel;
import org.openimaj.math.geometry.line.Line2d;
import org.openimaj.math.geometry.line.Line2d.IntersectionResult;
import org.openimaj.math.geometry.point.Point2d;
import org.openimaj.math.geometry.shape.Circle;
import org.openimaj.math.geometry.shape.Shape;

public class Quad {
	/* A class for storing a convex quadrilateral. */
	public float[] xs, ys;
	public float vertexRadius = 8;
	public static int INTERIOR = 5;
	public Integer selected; // index of selected vertex, or null, or INTERIOR
	public static float DEFAULT_PAD = 10, DEFAULT_WIDTH = 100, DEFAULT_HEIGHT = 100;
	
	public Quad() {
		xs = new float[4]; ys = new float[4]; // starting top-left, then clockwise.
		xs[0] = DEFAULT_PAD; xs[1] = DEFAULT_PAD + DEFAULT_WIDTH; xs[2] = xs[1]; xs[3] = xs[0];
		ys[0] = DEFAULT_PAD; ys[1] = DEFAULT_PAD; ys[2] = DEFAULT_PAD + DEFAULT_HEIGHT; ys[3] = ys[2];
	}
	
	public Pixel getXY(float u, float v) {
		/* u in [0,1] specifies horizontal extent from left to right
		 * v in [0,1] specifies vertical extent from top to bottom
		 * Return (x,y) location on the quad. */
		// Say point A lies u-way along 0->1,
		// B lies u-way along 3->2,
		// C lies v-way along 0->3
		// D lies v-way along 1->2
		// Then we want the intersection of AB and CD
			
		float
			xA = xs[0] + (xs[1]-xs[0])*u,
			yA = ys[0] + (ys[1]-ys[0])*u,
			xB = xs[3] + (xs[2]-xs[3])*u,
			yB = ys[3] + (ys[2]-ys[3])*u,
			xC = xs[0] + (xs[3]-xs[0])*v,
			yC = ys[0] + (ys[3]-ys[0])*v,
			xD = xs[1] + (xs[2]-xs[1])*v,
			yD = ys[1] + (ys[2]-ys[1])*v;
	
		float denom = (xA-xB)*(yC-yD) - (yA-yB)*(xC-xD);
		if (denom == 0f) {
			return new Pixel(0,0);
		}
		float px = ((xA*yB - yA*xB)*(xC-xD) - (xA-xB)*(xC*yD-yC*xD))/denom;
		float py = ((xA*yB - yA*xB)*(yC-yD)-(yA-yB)*(xC*yD-yC*xD))/denom;
		return new Pixel((int)px, (int)py);
	}
	
	public Shape getVertexShape(int k) {
		return new Circle(xs[k], ys[k], vertexRadius);
	}
	
	public boolean isInside(int x, int y) {
		/* Return true iff the point (x,y) is inside the quad. */
		double wind = 0.;
		for (int k=0; k < 4; ++k) {
			wind += subtend(x,y, xs[k], ys[k], xs[(k+1)%4], ys[(k+1)%4]);
		}
		if (wind > -0.01 && wind < 0.01) return false;
		return true;
	}
	
	public double subtend(float xo, float yo, float xa, float ya, float xb, float yb) {
		/* Return angle subtended from oa to ob, in [-pi, pi]. */
		double tb = Math.atan2(yb-yo, xb-xo);
		double ta = Math.atan2(ya-yo, xa-xo);
		double diff = (4.*Math.PI + tb-ta) % (2.*Math.PI); // always >= 0
		if (diff <= Math.PI) return diff;
		return diff - 2.*Math.PI;
	}
	
	public boolean isSelected(int x, int y, int vertex) {
		/* Return true iff (x,y) is "inside" the vertex-th vertex. */
		return getVertexShape(vertex).isInside(new Pixel(x,y));
	}
	public void updateSelected(int x, int y) {
		/* Update index of selected vertex based on mouse move to (x,y). */
		selected = null;
		for (int k = 0; k < 4; ++k) {if (isSelected(x,y,k)) {selected = k;}}
		if (selected == null && isInside(x,y)) selected = Quad.INTERIOR;
	}
	public void moveSelectedBy(int dx, int dy) {
		/* If there is a selected vertex, move it by (dx,dy).
		 * If the interior region is selected, move all vertices by (dx,dy). */
		if (selected==null) return;
		else if (selected == Quad.INTERIOR) {for (int k=0;k<4;++k) {xs[k]+=dx; ys[k] += dy;}}
		else {xs[selected] += dx; ys[selected] += dy;}
	}
	
//	public void moveSelectedTo(int x, int y, int width, int height) {
//		/* If there is a selected vertex, move it to the specified position.
//		 * If the interior region is selected, move the centroid to the specified position,
//		 * unless any vertex would go outside the bounds of (width, height). */
//		if (selected == null) return;
//		else if (selected == Quad.INTERIOR) {
//			float cx = 0f; float cy = 0f; for (int k=0; k<4; ++k) {cx += 0.25f*xs[k]; cy += 0.25f*ys[k];}
//			// got center position (cx, cy). Now move all verts to the new center, plus their difference from the original center
//			float newxs[] = new float[4]; float newys[] = new float[4];
//			for (int k = 0; k < 4; ++k) {
//				newxs[k] = x + xs[k] - cx;
//				newys[k] = y + ys[k] - cy;
//				if (newxs[k] < 0 || newxs[k] > width-1 || newys[k] < 0 || newys[k] > height-1) {
//					return;
//				}
//			}
//			xs = newxs; ys = newys;
//		}
//		else {xs[selected] = x; ys[selected] = y;}
//	}

	
	public void boundWithin(int width, int height) {
		/* Force all vertices inside the region (0,0) to (width,height). */
		for (int k = 0; k < 4; ++k) {
			xs[k] = Math.min(Math.max(xs[k],0),width-1);
			ys[k] = Math.min(Math.max(0, ys[k]), height-1);
		}
	}
}
