/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

package com.salelder;

import java.awt.Dimension;
import java.awt.event.MouseAdapter;
import java.io.File;
import java.io.IOException;
import java.util.List;

import javax.swing.JPanel;

import org.openimaj.image.FImage;
import org.openimaj.image.ImageUtilities;
import org.openimaj.image.MBFImage;
import org.openimaj.image.colour.RGBColour;
import org.openimaj.image.colour.Transforms;
import org.openimaj.image.pixel.Pixel;
import org.openimaj.image.processing.edges.CannyEdgeDetector;
import org.openimaj.image.processing.face.detection.DetectedFace;
import org.openimaj.image.processing.face.detection.FaceDetector;
import org.openimaj.image.processing.face.detection.HaarCascadeDetector;
import org.openimaj.image.processing.resize.ResizeProcessor;
import org.openimaj.image.processor.ImageProcessor;
import org.openimaj.image.processor.SinglebandImageProcessor;
import org.openimaj.math.geometry.shape.Circle;
import org.openimaj.math.geometry.shape.Rectangle;
import org.openimaj.math.geometry.shape.Shape;
import org.openimaj.video.Video;
import org.openimaj.video.VideoDisplay;
import org.openimaj.video.VideoDisplayListener;
import org.openimaj.video.capture.Device;
import org.openimaj.video.capture.VideoCapture;
import org.openimaj.video.capture.VideoCaptureException;

public class PreviewPanel extends JPanel {
	/* Maintains a reference image, quad, and reference filename. */
	public static int VIDEO_WIDTH = 640;
	public static int VIDEO_HEIGHT = 480;
	private static Float[] LINECOLOR = RGBColour.GREEN;
	private static Float[] ACTIVECOLOR = RGBColour.RED;
	
	//public static Double NOMINAL_COLOR_TEMP = 4000.; // in Kelvin
	// assume if a black body radiator at 4000 K hit a white page, the
	// camera would register that as (RGB) = (1,1,1)
	
	private CannyEdgeDetector cannyEdgeDetector;
	
	public Quad quad;
	private Video<MBFImage> video;
	private VideoDisplay<MBFImage> display;
	private File referenceFile;
	private MBFImage reference;
	private MBFImage referenceOriginalSize;
	private float opacity = 0.5f; // of reference on top of webcam video, in [0,1]
	public float brightness = 1f; // relative brightness of webcam video, in [0,2]
	public float contrast = 1f; // relative contrast of webcam video, in [0,2]
	public float scale = 1f; // amount to scale webcam video, in [0.5, 2]
	public float panX = 0f; // amount to move the webcam, relative to its width, in [0,1]
	public float panY = 0f; // " height
	public double colorTemp = 6500.0; // in Kelvin
	private double[] tempRGB = {1.0, 1.0, 1.0};
	private boolean distortCam = false; // if false, distort reference. if true, distort video.
	private boolean edgeDetect = false;
	private PreviewMouse mouse;
	private App app;
	
	public PreviewPanel(App a) {
		app = a;
		quad = new Quad();
		mouse = new PreviewMouse(this);
        setWebcam(null); // try to initialize with default camera
        cannyEdgeDetector = new CannyEdgeDetector();
	}
	
	public int getVideoWidth() {
		if (video == null) return VIDEO_WIDTH;
		else return video.getWidth();
	}
	public int getVideoHeight() {
		if (video == null) return VIDEO_HEIGHT;
		else return video.getHeight();
	}
	public void setReference(File f) throws IOException {
		if (f == null) {referenceFile = null; reference = null; return;}
		MBFImage ref = ImageUtilities.readMBF(f);
		referenceFile = f;
		setReferenceImage(ref);
	}
	public void setReferenceImageWithoutFile(MBFImage img) {
		/* Set an MBFImage as the reference image with no associated file.
		 * This means if the user saves the configuration, the file location will not be included. */
		referenceFile = null;
		setReferenceImage(img);
	}
	public File getReferenceFile() {
		return referenceFile;
	}
	
	public void setWebcam(Device d) {
		setWebcam(d, VIDEO_WIDTH, VIDEO_HEIGHT);
	}
	
	public void setVideoSize(int width, int height) {
		setWebcam(null, width, height);
	}
	
	public void setWebcam(Device d, int video_width, int video_height) {
		if (video != null) {
			video.close(); // Otherwise, cannot reopen since device is busy
			display.close(); // This is important! It will ``work'' without it, but after changing video size a few times,
				// you start to get bad lag. I think that's because the old video displays are still looking for
				// events in the background, and the garbage collector isn't fast enough to kill them.
				// With display.close(), this seems to have fixed that problem of latency after resizing a few times.
			this.remove(display.getScreen());}
		try {
			if (d==null) {
				d = VideoCapture.getVideoDevices().get(0);
			}
			video = new VideoCapture(video_width, video_height, d);
			resizeReferenceImage();
			quad.boundWithin(video.getWidth(), video.getHeight());
			
			display = VideoDisplay.createVideoDisplay(video, this);
	        display.addVideoListener(
	          new VideoDisplayListener<MBFImage>() {
	            public void beforeUpdate(MBFImage frame) {
	            	adjustFrameColor(frame);
	            	
	            	if (reference != null && !distortCam) {
	            		drawReference(frame);
	            		drawQuad(frame);
	            	}
	            	else if (reference != null && distortCam) {
	            		MBFImage clone = frame.clone();
	            		frame.fill(RGBColour.BLACK);
	            		MBFImage reference = PreviewPanel.this.reference;
	            		if (edgeDetect) {
	            			reference = PreviewPanel.this.reference.clone();
	            			drawEdges(reference, RGBColour.RED);}
	            		frame.drawImage(reference, 0,0);
	            		drawSelectedRegion(frame, clone);
	            	}
	            	
	            	transformFrame(frame);
	            }
	
	            public void afterUpdate(VideoDisplay<MBFImage> display) {
	            }
	          });
        	display.getScreen().addMouseListener(mouse); // for clicks
        	display.getScreen().addMouseMotionListener(mouse); // for moves
        	app.setStatus("Started webcam.");
        	app.revalidate(); // required when you remove, says Stack Overflow. Then see below, repaint req'd after revalidate
        	app.repaint(); // black magic. Need both revalidate and repaint to get rid of old video AND re-center new video in pane
        	// https://docs.oracle.com/javase/tutorial/uiswing/components/jcomponent.html#custompaintingapi says always invoke repaint
        	// after revalidate
		} catch (Exception e) {
			//video = null;
			app.setStatus("Error: Couldn't start webcam.");
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		//repaint();
	}
	
	private void setReferenceImage(MBFImage reference) {
		/* Set the reference image and resize it to fit inside the webcam image. */
		this.referenceOriginalSize = reference.clone();
		this.reference = reference;
		float scale = Math.min(
				1f*video.getHeight() / reference.getHeight(),
				1f*video.getWidth() / reference.getWidth());
		this.reference = reference.process(new ResizeProcessor(scale)); // rescale to fit the display
		//reference = myScaleImage(reference, scale);
	}
	
	private void resizeReferenceImage() {
		/* scale the current reference to fit inside the current webcam image. */
		if (referenceOriginalSize == null) return;
		setReferenceImage(referenceOriginalSize);
		
	}
	public void setOpacity(float o) {
		opacity = o;
	}
	public float getOpacity() {
		return opacity;
	}
	public boolean hasReference() {
		return referenceFile != null;
	}
	public void toggleView() {
		if (reference == null) return;
		distortCam = !distortCam;
	}
	
	private void drawQuad(MBFImage frame) {
		/* Draw this.quad onto frame. */
		for (int k = 0; k < 4; ++k) {
			// draw handle for kth vertex
			Shape s = quad.getVertexShape(k);
			Float[] vColor = LINECOLOR; Float[] lColor = LINECOLOR;
			if (quad.selected != null && quad.selected.intValue() == k) {
				vColor = ACTIVECOLOR;
			}
			frame.drawShape(s, vColor);
			if (quad.selected != null && quad.selected == Quad.INTERIOR) {lColor = ACTIVECOLOR;}
			// draw line connecting k-th vertex to next vertex in clockwise direction
			frame.drawLine((int)quad.xs[k], (int)quad.ys[k],
					(int)quad.xs[(k+1) % 4], (int)quad.ys[(k+1)%4], lColor);
		}
	}
	
	private void drawReference(MBFImage frame) {
		/* Draw the reference image onto the video frame. */
		//frame.drawPoint(quad.getXY(-0.5f, 0.f), ACTIVECOLOR, 10);
		MBFImage clone = frame.clone();
		//if (reference == null) return;
		// loop over image coordinates and display each image pixel at the corresponding
		// x,y location inside quad
		MBFImage reference = this.reference.clone(); // prevents bounds error if reference is changed during loop
		if (edgeDetect) {drawEdges(reference, RGBColour.RED);}
		for (int row = 0; row < reference.getHeight(); ++row) {
			for (int col = 0; col < reference.getWidth(); ++col) {
				float u = 1f * col / reference.getWidth();
				float v = 1f * row / reference.getHeight();
				Pixel destination = quad.getXY(u,v);
				boundWithin(destination, frame);
				Float[] camColor = getPixel(clone,destination);
				Float[] refColor = getPixel(reference, col, row);
				Float[] mixColor = mixColor(camColor, refColor, opacity);
				if (edgeDetect) {mixColor = addColor(camColor, refColor);}
				frame.drawPoint(destination, mixColor, 1);
			}
		}	
	}
	
	public void updateSelected(int x, int y) {
		/* If (x,y) is the mouse location in transformed coordinates, update quad selection. */
		// Convert to untransformed coordinates and give those to the quad.
		int width = VIDEO_WIDTH; int height = VIDEO_HEIGHT;
		if (video != null) {width = video.getWidth(); height = video.getHeight();}
		Pixel old_loc = transformedToUntransformedCoordinates(width, height, x, y);
		if (quad != null) {quad.updateSelected(old_loc.x, old_loc.y);}
		
	}
	
	private Pixel transformedToUntransformedCoordinates(int width, int height, int x, int y) {
		/* For a location (x,y) on the screen after transformation has been applied,
		 * convert to (u,v) of the original untransformed image corresponding to that point. */
		int old_row = (int) (0.5f*height + 1f/scale * (y - height * (0.5 - panY)));
		int old_col = (int) (0.5f*width + 1f/scale * (x - width * (0.5 + panX)));
		return new Pixel(old_col, old_row);
	}

	private void transformFrame(MBFImage frame) {
		/* Apply scale and pan operations to frame. */
		MBFImage original = frame.clone();
		for (int r=0; r < original.getRows(); ++r) {
			for (int c=0; c < original.getCols(); ++c) {
				// c,r is the location we're going to write to
				Pixel old_loc = transformedToUntransformedCoordinates(original.getCols(), original.getRows(), c, r);
				Float[] color = getPixel(original, old_loc.x,old_loc.y);
				
				frame.drawPoint(new Pixel(c,r), color, 1);
			}
		}
		if (edgeDetect) {drawEdges(frame, RGBColour.BLUE); }//frame.getBand(0).fill(0); frame.getBand(1).fill(0);}
	}
	
	private void adjustFrameColor(MBFImage frame) {
		/* Apply brightness, contrast, and white balance adjustments to frame. */
		MBFImage original = frame.clone();
		for (int r=0; r<original.getRows(); ++r) {
			for (int c=0; c<original.getCols(); ++c) {
				Float[] color = getPixel(original, c,r);
				for(int band =0; band<3; ++band) {
					color[band] /= (float)tempRGB[band];
					color[band] = sigmoid(color[band], contrast);
					color[band] *= brightness;
					color[band]=Math.min(1f, color[band]); // clamp to 1
				}
				frame.drawPoint(new Pixel(c,r), color, 1);
			}
		}
		
	}
	
	private void drawEdges(MBFImage frame, Float[] color) {
		/* Draw the edges of frame in the color provided. */
		frame.processInplace(cannyEdgeDetector);
		for (int r=0; r<frame.getHeight();++r) {
			for (int c=0; c<frame.getWidth(); ++c) {
				Float[] col=frame.getPixel(c,r);
				if (col[0]>0.5 && col[1]>0.5 && col[2] > 0.5) {frame.setPixel(c, r, color);}
				else {frame.setPixel(c, r, RGBColour.BLACK);}
			}
		}
		
	}

	private float sigmoid(float x, float p) {
		/* Map x in [0,1] to [0,1]. p is a parameter in [0, +infinity) controlling the contrast.
		 * If p=0., always return 0.5. If p=1, return x. If p large, return values close to 0 or 1. */
		if (x<=0.5) {return (float) (Math.pow(x, p) * Math.pow(0.5, 1f-p));}
		return 1f - sigmoid(1f - x, p);
	}
	
	private Float[] getPixel(MBFImage im, int x, int y) {
		/* Bounds-check the location against the image dimensions and return Black if not in bounds. */
		if (x<im.getCols() && x>=0 && y<im.getRows() && y>=0) {
			return im.getPixel(x,y);
		}
		return RGBColour.BLACK;
	}
	private Float[] getPixel(MBFImage im, Pixel p) {
		return getPixel(im, (int)p.getX(), (int)p.getY());
	}
	
	private void drawSelectedRegion(MBFImage frame, MBFImage image) {
		/* Take the part of image within quad, and draw it on frame, within the
		 * span of reference. */
		// For each row,col of reference, look up the location on the quad
		// to pull from.
		MBFImage reference = this.reference.clone();
		
		for (int row = 0; row < reference.getHeight(); ++row) {
			for (int col = 0; col < reference.getWidth(); ++col) {
				Pixel quadLocation =
						quad.getXY(
								1f*col/reference.getWidth(),
								1f*row/reference.getHeight());
				Float[] refColor = getPixel(frame, col,row);
				Float[] camColor = getPixel(image, quadLocation);
				Float[] newColor = mixColor(camColor, refColor, opacity);
				if (edgeDetect) {newColor = addColor(camColor, refColor);}
				frame.drawPoint(new Pixel(col,row), newColor, 1);
			}
		}
	}
	
	private Float[] mixColor(Float[] color1, Float[] color2, float opacity) {
		/* Overlay color2 with opacity on top of color1. */
		Float[] res = new Float[3];
		for (int k=0; k < 3; ++k) {res[k] = (1f-opacity)*color1[k] + opacity*color2[k];}
		return res;
	}
	
	private Float[] addColor(Float[] color1, Float[] color2) {
		/* Add color1 and color2 and clip if necessary. */
		Float[] res = new Float[3];
		for (int k=0; k < 3; ++k) {res[k] = Math.min(1f, color1[k]+color2[k]);}
		return res;
	}
	
	private void boundWithin(Pixel px, MBFImage img) {
		/* Force the pixel to be inside the image bounds. */
		if (px.getX() >= img.getWidth()-1) {px.setX(img.getWidth() - 1);}
		if (px.getX() < 0) {px.setX(0);}
		if (px.getY() >= img.getHeight()-1) {px.setY(img.getHeight() - 1);}
		if (px.getY() < 0) {px.setY(0);}
	}
	
	public Dimension getPreferredSize() {
		return new Dimension(VIDEO_WIDTH, VIDEO_HEIGHT);
	}

	public void setBrightness(float brightness) {
		this.brightness = brightness;
	}

	public void setContrast(float contrast) {
		// TODO Auto-generated method stub
		this.contrast = contrast;
	}

	public void setScale(float scale) {
		// TODO Auto-generated method stub
		this.scale = scale;
	}

	public void setPanX(float panX) {
		// TODO Auto-generated method stub
		this.panX= panX;
	}

	public void setPanY(float panY) {
		// TODO Auto-generated method stub
		this.panY = panY;
	}
	
	public void setColorTemp(double temp) {
		colorTemp = temp;
		tempRGB = Transforms.kelvinToRGB(temp);
	}

	public void toggleEdges() {
		/* Turn edge detection on or off. */
		edgeDetect = !edgeDetect;
	}

}
