/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

package com.salelder;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.Signature;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Enumeration;

import org.json.JSONObject;
import org.json.JSONTokener;

import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.hardware.ComputerSystem;
import oshi.hardware.HardwareAbstractionLayer;
import oshi.software.os.OperatingSystem;

import java.net.http.HttpRequest.BodyPublishers;

public class License {
	/* Represents a software license which may or may not contain a license key or be activated.
	 * Note: The current version is free and open-source, so most of this code is bypassed or not used. */
	public static int SUCCESS = 0, MACHINE_ALREADY_REGISTERED = 1, LICENSE_NOT_RECOGNIZED = 2, CONNECTION_ERROR = 3,
			UNEXPECTED_RESPONSE = 4, MAX_DEVICES_REGISTERED = 5;
	private static String LICENSE_FILENAME = "license.json";
	
	private String license, signature;
	private boolean active;
	
	public License() {
		/* Attempt to load license key and signature from a file. */
		active = false;
		//loadLicense();
	}
	
	public License(String l) {
		/* Create a license from a key. */
		active = false;
		license = l;
	}
	
	private String getMachineID() {
		/* Return a string identifying this machine. May not be unique, but should be close. */
		SystemInfo si = new SystemInfo();
		HardwareAbstractionLayer hardware = si.getHardware();
		String model = hardware.getComputerSystem().getModel();
		String uuid = si.getHardware().getComputerSystem().getHardwareUUID();
		String os = System.getProperty("os.name");
		String uname = System.getProperty("user.name");
		return String.format("%s-%s-%s-%s", uname,os,model,uuid);
	}
	
	public static byte[] hexStringToBytes(String hexString) {
		/* Return the byte array encoded by the given hex string.
		 * E.g. "f012" will be converted to [0xf0, 0x12] */
		byte[] result = new byte[hexString.length() / 2];
		for (int k = 0; k < hexString.length()/2; ++k) {
			//result[k] = Byte.parseByte(hexString.substring(2*k,2*k+2), 16); parseByte thinks it's signed or something
			result[k]= (byte) Integer.parseInt(hexString.substring(2*k,2*k+2), 16);
		}
		return result;
	}
	
	public void setLicense(String l) {
		/* Add the license key but do not activate it. */
		license = l;
		active = false;
	}
	
	public boolean isActive() {
		return true;
		//return active;
	}
	
	private boolean isSignatureValid() {
		if (signature == null) return false;
		try {
		/* Return true iff this.signature is the signed machine ID. */
		// resources folder already added to classpath, so don't prefix with resources/
		InputStream s= getClass().getClassLoader().getResourceAsStream("signature-public-key.txt");
		
		BufferedReader br = new BufferedReader(new InputStreamReader(s));
		String nextLine = br.readLine(); // doesn't include \n
		ArrayList<String> lines = new ArrayList<String>();
		while (nextLine != null) {
			lines.add(nextLine);
			nextLine = br.readLine();
		}
		StringBuilder sb = new StringBuilder();
		for (int k=1; k < lines.size()-1; ++k) {sb.append(lines.get(k));}
		String readableKey = sb.toString();
		byte[] binaryKey = Base64.getDecoder().decode(readableKey);
		X509EncodedKeySpec keySpec = new X509EncodedKeySpec(binaryKey);
		PublicKey key = KeyFactory.getInstance("RSA").generatePublic(keySpec);
		
		Signature sig = Signature.getInstance("SHA256withRSA");
		sig.initVerify(key);
		sig.update(getMachineID().getBytes());		
		byte[] signatureBytes = hexStringToBytes(signature);
		return sig.verify(signatureBytes);
		} catch (Exception e) {return false;}
	}
	
	public int activate() {
		/* Send license and machine id to server and attempt to store the signature returned by the server.
		 * Return status code License.SUCCESS, MACHINE_ALREADY_REGISTERED, or LICENCE_NOT_RECOGNIZED */
		HttpClient client = HttpClient.newHttpClient();
		String postdata = String.format("license=%s&machine=%s", license, URLEncoder.encode(getMachineID(), StandardCharsets.UTF_8));
		HttpRequest request = HttpRequest.newBuilder()
				.uri(URI.create("https://salelder.com/activation/activate.php"))
				.POST(BodyPublishers.ofString(postdata))
				//.header("Content-Type", "application/json")
				.header("Content-Type", "application/x-www-form-urlencoded") // Need this to recognize post in format above
				.build();
		HttpResponse<String> response; JSONObject result;
		try {
			response = client.send(request, BodyHandlers.ofString());
			
			result = new JSONObject(response.body());
		} catch (Exception e) {
			return CONNECTION_ERROR;
		}
		try{
			String status = result.get("status").toString();
			if (status.equals("LICENSE_NOT_RECOGNIZED")) return LICENSE_NOT_RECOGNIZED;
			if (status.equals("MAX_DEVICES_REGISTERED")) return MAX_DEVICES_REGISTERED;
			signature = result.get("signature").toString();
			active = isSignatureValid();
			saveLicense();
			
			if (status.equals("MACHINE_ALREADY_REGISTERED")) return MACHINE_ALREADY_REGISTERED;
			return SUCCESS;
		
		} catch (Exception e) {return UNEXPECTED_RESPONSE;}
		
	}
	private void saveLicense() {
		/* Attempt to save license and signature to a json file. */
		try {
			JSONObject record = new JSONObject();
			record.put("license", license).put("signature", signature);
			System.out.println(record.toString());
			File f = new File(LICENSE_FILENAME);
			FileWriter writer = new FileWriter(f);
			writer.write(record.toString());
			writer.close();
		} catch (Exception e) {return;}
	}
	private void loadLicense() {
		/* Attempt to load signature and license from a json file. */
		try {
			FileReader reader = new FileReader(LICENSE_FILENAME);
			JSONTokener tokener = new JSONTokener(reader);
			JSONObject record = new JSONObject(tokener);
			license = record.getString("license");
			signature = record.getString("signature");
		} catch (Exception e) {return;}
		active = isSignatureValid();
	}
}
