/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

package com.salelder;

import java.awt.BorderLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSplitPane;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.openimaj.image.DisplayUtilities;
import org.openimaj.image.FImage;
import org.openimaj.image.ImageUtilities;
import org.openimaj.image.MBFImage;
import org.openimaj.image.colour.ColourSpace;
import org.openimaj.image.colour.RGBColour;
import org.openimaj.image.colour.Transforms;
import org.openimaj.image.processing.convolution.FGaussianConvolve;
import org.openimaj.image.processing.edges.CannyEdgeDetector;
import org.openimaj.image.processing.face.detection.DetectedFace;
import org.openimaj.image.processing.face.detection.FaceDetector;
import org.openimaj.image.processing.face.detection.HaarCascadeDetector;
import org.openimaj.image.typography.hershey.HersheyFont;
import org.openimaj.video.Video;
import org.openimaj.video.VideoDisplay;
import org.openimaj.video.VideoDisplayListener;
import org.openimaj.video.capture.Device;
import org.openimaj.video.capture.VideoCapture;

/**
 * OpenIMAJ Hello world!
 *
 */
public class App extends JFrame implements FileUser {
	public static String APP_NAME= "Vermeer Live";
	public static String VERSION= "1.0";
	
	public JFileChooser fileChooser;
	public JMenuBar menubar;
	private PreviewPanel preview;
	private OptionsPanel options;
	private JLabel statusMessage;
	private File activeFile;
	private PreviewController previewController;
	private License license;
	
	private JMenuItem saveItem, saveAsItem, openItem;
	
	public App() {
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setTitle(APP_NAME);
		
		//license = new License("abc123");
		//license.activate();
		/* Note: In previous versions I added licensing, but now it's free and open-source.
		 * I kept the license code in place for reference, but I've just made the License
		 * class always report that it's activated. */
		license = new License();
		if (!license.isActive()) {
			JOptionPane.showMessageDialog(this,
					"This is a demo version. Some features are missing or incomplete.\nTo run the full version, please purchase a license key.",
				"Demo Version", JOptionPane.INFORMATION_MESSAGE);
		}
		
		menubar = new JMenuBar();
		JMenu filemenu = new JMenu("File"); menubar.add(filemenu); filemenu.setMnemonic('f');
		JMenuItem newItem = new JMenuItem("New"); filemenu.add(newItem);
		openItem = new JMenuItem("Open"); filemenu.add(openItem);
		filemenu.addSeparator();
		saveItem = new JMenuItem("Save"); filemenu.add(saveItem);
		saveAsItem = new JMenuItem("Save As"); filemenu.add(saveAsItem);
		int ctrlKey = Toolkit.getDefaultToolkit().getMenuShortcutKeyMaskEx();
		KeyStroke ctrlN = KeyStroke.getKeyStroke(KeyEvent.VK_N, ctrlKey);
		KeyStroke ctrlO = KeyStroke.getKeyStroke(KeyEvent.VK_O, ctrlKey);
		KeyStroke ctrlS = KeyStroke.getKeyStroke(KeyEvent.VK_S, ctrlKey);
		newItem.setAccelerator(ctrlN);
		openItem.setAccelerator(ctrlO);
		saveItem.setAccelerator(ctrlS);
		FileNameExtensionFilter jsonFilter = new FileNameExtensionFilter("json", "json");	
		fileChooser = new JFileChooser();
		FileListener openListener = new FileListener(jsonFilter, fileChooser, menubar,
				FileListener.OPEN_DIALOG, this);
		FileListener saveListener = new FileListener(jsonFilter, fileChooser, menubar,
				FileListener.SAVE_DIALOG, this);
		FileListener saveAsListener = new FileListener(jsonFilter, fileChooser, menubar,
				FileListener.SAVE_AS_DIALOG, this);
		openItem.addActionListener(openListener);
		saveItem.addActionListener(saveListener);
		saveAsItem.addActionListener(saveAsListener);
				
		JMenu viewmenu = new JMenu("View"); menubar.add(viewmenu); viewmenu.setMnemonic('v');
		JMenuItem toggleImageItem = new JMenuItem("Switch image");
		viewmenu.add(toggleImageItem);
		toggleImageItem.setAccelerator(KeyStroke.getKeyStroke('q'));
		
		JMenu helpmenu = new JMenu("Help"); menubar.add(helpmenu); helpmenu.setMnemonic('h');
		//JMenuItem registerItem = new JMenuItem("Activate product"); helpmenu.add(registerItem);
		helpmenu.addSeparator();
		//JMenuItem tutorialItem = new JMenuItem("Manual"); helpmenu.add(tutorialItem);
		JMenuItem aboutItem = new JMenuItem("About"); helpmenu.add(aboutItem);
		
		//registerItem.addActionListener(new LicenseDialog(this, license));
		aboutItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent arg0) {
				JOptionPane.showMessageDialog(App.this,
						String.format("%s\nVersion %s\nCopyright \u00a9 2021 Sal Elder\nReleased under an MIT license.", APP_NAME, VERSION));
			}
			
		});
					
		JPanel statusPanel = new JPanel();
		statusMessage = new JLabel("");
		statusPanel.add(statusMessage);
		
		preview = new PreviewPanel(this);
		options = new OptionsPanel(preview,license);
		//previewController = options.getPreviewController();
		
		toggleImageItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent arg0) {
				options.toggleOpacity();
			}
		});
		
		notifyLicenseChanged(); // enable or disable the save/open options
		
		setJMenuBar(menubar);
		//add(preview, BorderLayout.WEST);
		//add(options, BorderLayout.EAST);
		JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
				preview, options);
		add(split, BorderLayout.CENTER);
		add(statusPanel, BorderLayout.PAGE_END);
		
		pack();
		setVisible(true);
	}
	
	public void setStatus(String message) {
		statusMessage.setText(message);
	}
	
	public void notifyLicenseChanged() {
		/* Update GUI elements to reflect new license activation status. */
		boolean active = license.isActive();
		saveItem.setEnabled(active);
		saveAsItem.setEnabled(active);
		openItem.setEnabled(active);
	}

    public static void main( String[] args ) throws IOException {
    	//System.out.println(Byte.toString((byte)255));
    	new App();
    	
    	//Create an image
        //MBFImage image = new MBFImage(1000,70, ColourSpace.RGB);
    	//MBFImage image = ImageUtilities.readMBF(new URL("http://static.openimaj.org/media/tutorial/sinaface.jpg"));
        //System.out.println(image.colourSpace);
        
               
        /*MBFImage clone = image.clone();
        for (int r = 0; r < clone.getHeight(); ++r) {
        	for (int c=0; c < clone.getWidth(); ++c) {
        		float val = clone.getBand(0).pixels[r][c];
        		clone.getBand(0).pixels[r][c] = (float) Math.pow(Math.sin(val * Math.PI/ 2.), 2);
        	}
        }*/
    	//Fill the image with white
        //image.fill(RGBColour.WHITE);
        		        
        //Render some test into the image
        //image.drawText("Hello World, how are you?", 10, 60, HersheyFont.CURSIVE, 50, RGBColour.BLACK);

        //Apply a Gaussian blur
        //image.processInplace(new FGaussianConvolve(2f));
        
        //Display the image
        //DisplayUtilities.display(clone);//.getBand(0), "Red channel");
    }

	@Override
	public boolean hasActiveFile() {
		return activeFile != null;
	}

	@Override
	public void openFile(File f) {
		try {
			ConfigFile config = new ConfigFile(f);
			preview.quad = config.getQuad();
			preview.setReference(config.getReference());
			options.setParams(config.getBrightness(), config.getContrast(), config.getScale(),
					config.getPanX(), config.getPanY(), config.getTemp(), config.getVideoWidth(), config.getVideoHeight());
		} catch (IOException e) {
			e.printStackTrace();
			setStatus("Error: couldn't load settings.");
		}
		
	}

	@Override
	public void saveFile(File f) {
		ConfigFile config = new ConfigFile(
				preview.quad, preview.getReferenceFile(), preview.brightness,
				preview.contrast, preview.scale, preview.panX, preview.panY,
				preview.colorTemp,
				preview.getVideoWidth(), preview.getVideoHeight());
		try {
			config.saveJSON(f);
			activeFile = f;
			setStatus("Saved settings.");
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			setStatus("Error: couldn't save settings.");
		}
	}

	@Override
	public void saveFile() {
		saveFile(activeFile);		
	}
}
