/*
 * Decompiled with CFR 0.152.
 */
package org.openimaj.ml.kernel;

import org.openimaj.citation.annotation.Reference;
import org.openimaj.citation.annotation.ReferenceType;
import org.openimaj.citation.annotation.References;
import org.openimaj.feature.DoubleFV;
import org.openimaj.feature.FeatureExtractor;
import org.openimaj.feature.FeatureVector;
import org.openimaj.math.util.MathUtils;

@References(references={@Reference(type=ReferenceType.Article, author={"Vedaldi, A.", "Zisserman, A."}, title="Efficient Additive Kernels via Explicit Feature Maps", year="2012", journal="Pattern Analysis and Machine Intelligence, IEEE Transactions on", pages={"480", "492"}, number="3", volume="34", customData={"keywords", "approximation theory;computer vision;data handling;feature extraction;learning (artificial intelligence);spectral analysis;support vector machines;Nystrom approximation;additive homogeneous kernels;approximate finite-dimensional feature maps;approximation error;computer vision;data dependency;explicit feature maps;exponential decay;large scale nonlinear support vector machines;linear SVM;spectral analysis;Additives;Approximation methods;Histograms;Kernel;Measurement;Support vector machines;Training;Kernel methods;feature map;large scale learning;object detection.;object recognition", "doi", "10.1109/TPAMI.2011.153", "ISSN", "0162-8828"}), @Reference(type=ReferenceType.Inproceedings, author={"A. Vedaldi", "A. Zisserman"}, title="Efficient Additive Kernels via Explicit Feature Maps", year="2010", booktitle="Proceedings of the IEEE Conf. on Computer Vision and Pattern Recognition (CVPR)")})
public class HomogeneousKernelMap {
    private KernelType kernelType;
    private double period;
    private double gamma;
    private int order;
    private int numSubdivisions;
    private double subdivision;
    private int minExponent;
    private int maxExponent;
    private double[] table;

    public HomogeneousKernelMap(KernelType kernelType, WindowType windowType) {
        this(kernelType, 1.0, 1, -1.0, windowType);
    }

    public HomogeneousKernelMap(KernelType kernelType, double gamma, WindowType windowType) {
        this(kernelType, gamma, 1, -1.0, windowType);
    }

    public HomogeneousKernelMap(KernelType kernelType, double gamma, int order, WindowType windowType) {
        this(kernelType, gamma, order, -1.0, windowType);
    }

    public HomogeneousKernelMap(KernelType kernelType, double gamma, int order, double period, WindowType windowType) {
        if (gamma <= 0.0) {
            throw new IllegalArgumentException("Gamma must be > 0");
        }
        this.kernelType = kernelType;
        this.gamma = gamma;
        this.order = order;
        if (period < 0.0) {
            period = this.computeDefaultPeriod(windowType, kernelType);
        }
        this.period = period;
        this.numSubdivisions = 8 + 8 * order;
        this.subdivision = 1.0 / (double)this.numSubdivisions;
        this.minExponent = -20;
        this.maxExponent = 8;
        int tableHeight = 2 * this.order + 1;
        int tableWidth = this.numSubdivisions * (this.maxExponent - this.minExponent + 1);
        this.table = new double[tableHeight * tableWidth + 2 * (1 + this.order)];
        int tableOffset = 0;
        int kappaOffset = tableHeight * tableWidth;
        int freqOffset = kappaOffset + 1 + order;
        double L = Math.PI * 2 / this.period;
        int j = 0;
        int i = 0;
        while (i <= this.order) {
            this.table[freqOffset + i] = j;
            this.table[kappaOffset + i] = windowType.getSmoothSpectrum((double)j * L, this);
            if (!(this.table[kappaOffset + i] > 0.0) && ++j < 3 * i) continue;
            ++i;
        }
        for (int exponent = this.minExponent; exponent <= this.maxExponent; ++exponent) {
            double mantissa = 1.0;
            i = 0;
            while (i < this.numSubdivisions) {
                double x = mantissa * Math.pow(2.0, exponent);
                double xgamma = Math.pow(x, this.gamma);
                double Lxgamma = L * xgamma;
                double Llogx = L * Math.log(x);
                this.table[tableOffset++] = Math.sqrt(Lxgamma * this.table[kappaOffset]);
                for (j = 1; j <= this.order; ++j) {
                    double sqrt2kappaLxgamma = Math.sqrt(2.0 * Lxgamma * this.table[kappaOffset + j]);
                    this.table[tableOffset++] = sqrt2kappaLxgamma * Math.cos(this.table[freqOffset + j] * Llogx);
                    this.table[tableOffset++] = sqrt2kappaLxgamma * Math.sin(this.table[freqOffset + j] * Llogx);
                }
                ++i;
                mantissa += this.subdivision;
            }
        }
    }

    private double computeDefaultPeriod(WindowType windowType, KernelType kernelType) {
        double period = 0.0;
        block0 : switch (windowType) {
            case Uniform: {
                switch (kernelType) {
                    case Chi2: {
                        period = 5.86 * Math.sqrt(this.order + 0) + 3.65;
                        break;
                    }
                    case JensonShannon: {
                        period = 6.64 * Math.sqrt(this.order + 0) + 7.24;
                        break;
                    }
                    case Intersection: {
                        period = 2.38 * Math.log((double)this.order + 0.8) + 5.6;
                    }
                }
                break;
            }
            case Rectangular: {
                switch (kernelType) {
                    case Chi2: {
                        period = 8.8 * Math.sqrt((double)this.order + 4.44) - 12.6;
                        break block0;
                    }
                    case JensonShannon: {
                        period = 9.63 * Math.sqrt((double)this.order + 1.0) - 2.93;
                        break block0;
                    }
                    case Intersection: {
                        period = 2.0 * Math.log((double)this.order + 0.99) + 3.52;
                    }
                }
            }
        }
        return Math.max(period, 1.0);
    }

    public void evaluate(double[] destination, int stride, int offset, double x) {
        double sign;
        MathUtils.ExponentAndMantissa em = MathUtils.frexp((double)x);
        double mantissa = em.mantissa;
        int exponent = em.exponent;
        if ((mantissa *= 2.0 * (sign = mantissa >= 0.0 ? 1.0 : -1.0)) == 0.0 || --exponent <= this.minExponent || exponent >= this.maxExponent) {
            for (int j = 0; j <= this.order; ++j) {
                destination[offset + j * stride] = 0.0;
            }
            return;
        }
        int featureDimension = 2 * this.order + 1;
        int v1offset = (exponent - this.minExponent) * this.numSubdivisions * featureDimension;
        mantissa -= 1.0;
        while (mantissa >= this.subdivision) {
            mantissa -= this.subdivision;
            v1offset += featureDimension;
        }
        int v2offset = v1offset + featureDimension;
        for (int j = 0; j < featureDimension; ++j) {
            double f1 = this.table[v1offset++];
            double f2 = this.table[v2offset++];
            destination[offset + j * stride] = sign * ((f2 - f1) * ((double)this.numSubdivisions * mantissa) + f1);
        }
    }

    public DoubleFV evaluate(DoubleFV in) {
        int step = 2 * this.order + 1;
        DoubleFV out = new DoubleFV(step * in.length());
        for (int i = 0; i < in.length(); ++i) {
            this.evaluate((double[])out.values, 1, i * step, ((double[])in.values)[i]);
        }
        return out;
    }

    public <T> FeatureExtractor<DoubleFV, T> createWrappedExtractor(FeatureExtractor<? extends FeatureVector, T> inner) {
        return new ExtractorWrapper<T>(inner, this);
    }

    public static class ExtractorWrapper<T>
    implements FeatureExtractor<DoubleFV, T> {
        private FeatureExtractor<? extends FeatureVector, T> inner;
        private HomogeneousKernelMap map;

        public ExtractorWrapper(FeatureExtractor<? extends FeatureVector, T> inner, HomogeneousKernelMap map) {
            this.inner = inner;
            this.map = map;
        }

        public DoubleFV extractFeature(T object) {
            return this.map.evaluate(((FeatureVector)this.inner.extractFeature(object)).asDoubleFV());
        }
    }

    public static enum WindowType {
        Uniform{

            @Override
            double computeKappaHat(KernelType type, double omega, HomogeneousKernelMap map) {
                return type.getSpectrum(omega);
            }
        }
        ,
        Rectangular{

            @Override
            double computeKappaHat(KernelType type, double omega, HomogeneousKernelMap map) {
                double kappa_hat = 0.0;
                double epsilon = 0.01;
                double omegaRange = 2.0 / (map.period * 0.01);
                double domega = 2.0 * omegaRange / 2049.0;
                for (double omegap = -omegaRange; omegap <= omegaRange; omegap += domega) {
                    double win = this.sinc(map.period / 2.0 * omegap);
                    kappa_hat += (win *= map.period / (Math.PI * 2)) * type.getSpectrum(omegap + omega);
                }
                kappa_hat *= domega;
                kappa_hat = Math.max(kappa_hat, 0.0);
                return kappa_hat;
            }

            private double sinc(double x) {
                if (x == 0.0) {
                    return 1.0;
                }
                return Math.sin(x) / x;
            }
        };


        abstract double computeKappaHat(KernelType var1, double var2, HomogeneousKernelMap var4);

        protected double getSmoothSpectrum(double omega, HomogeneousKernelMap map) {
            return this.computeKappaHat(map.kernelType, omega, map);
        }
    }

    public static enum KernelType {
        Intersection{

            @Override
            protected double getSpectrum(double omega) {
                return 0.6366197723675814 / (1.0 + 4.0 * omega * omega);
            }
        }
        ,
        Chi2{

            @Override
            protected double getSpectrum(double omega) {
                return 2.0 / (Math.exp(Math.PI * omega) + Math.exp(-Math.PI * omega));
            }
        }
        ,
        JensonShannon{

            @Override
            protected double getSpectrum(double omega) {
                return 2.0 / Math.log(4.0) * 2.0 / (Math.exp(Math.PI * omega) + Math.exp(-Math.PI * omega)) / (1.0 + 4.0 * omega * omega);
            }
        };


        protected abstract double getSpectrum(double var1);
    }
}

