/*
 * Decompiled with CFR 0.152.
 */
package gov.sandia.cognition.math;

import gov.sandia.cognition.annotation.CodeReview;
import gov.sandia.cognition.annotation.PublicationReference;
import gov.sandia.cognition.annotation.PublicationReferences;
import gov.sandia.cognition.annotation.PublicationType;
import gov.sandia.cognition.math.LentzMethod;
import gov.sandia.cognition.math.OperationNotConvergedException;
import gov.sandia.cognition.math.matrix.Vector;

@CodeReview(reviewer={"Kevin R. Dixon"}, date="2008-02-26", changesNeeded=false, comments={"Minor changes, log2 uses callback to main log() method.", "Otherwise, looks fine."})
public class MathUtil {
    public static double log(double x, double base) {
        return Math.log(x) / Math.log(base);
    }

    public static double log2(double x) {
        return MathUtil.log(x, 2.0);
    }

    @PublicationReferences(references={@PublicationReference(author={"Wikipedia"}, title="Gamma Function", type=PublicationType.WebPage, year=2010, url="http://en.wikipedia.org/wiki/Gamma_function"), @PublicationReference(author={"jdhedden"}, title="Bug in 2nd edition version of gammln()", type=PublicationType.WebPage, year=2005, url="http://www.numerical-recipes.com/forum/showthread.php?t=606")})
    public static double logGammaFunction(double input) {
        if (input <= 0.0) {
            throw new IllegalArgumentException("Input must be > 0.0");
        }
        double xx = input;
        double tmp = xx + 4.5;
        tmp -= (xx - 0.5) * Math.log(tmp);
        double ser = 1.000000000190015 + 76.18009172947146 / xx - 86.50532032941678 / (xx + 1.0) + 24.01409824083091 / (xx + 2.0) - 1.231739572450155 / (xx + 3.0) + 0.001208650973866179 / (xx + 4.0) - 5.395239384953E-6 / (xx + 5.0);
        return Math.log(2.5066282746310007 * ser) - tmp;
    }

    @PublicationReferences(references={@PublicationReference(author={"Wikipedia"}, title="Incomplete gamma function", type=PublicationType.WebPage, year=2010, url="http://en.wikipedia.org/wiki/Incomplete_gamma_function"), @PublicationReference(author={"William H. Press", "Saul A. Teukolsky", "William T. Vetterling", "Brian P. Flannery"}, title="Numerical Recipes in C, Second Edition", type=PublicationType.Book, year=1992, pages={218}, notes={"Function gammap"}, url="http://www.nrbook.com/a/bookcpdf.php")})
    public static double lowerIncompleteGammaFunction(double a, double x) {
        if (a <= 0.0) {
            throw new IllegalArgumentException("a must be > 0.0");
        }
        double gammp = x <= 0.0 ? 0.0 : (x > 1.0E10 ? 1.0 : (x < a + 1.0 ? MathUtil.incompleteGammaSeriesExpansion(a, x) : MathUtil.incompleteGammaContinuedFraction(a, x)));
        return gammp;
    }

    @PublicationReference(author={"William H. Press", "Saul A. Teukolsky", "William T. Vetterling", "Brian P. Flannery"}, title="Numerical Recipes in C, Second Edition", type=PublicationType.Book, year=1992, pages={218, 219}, url="http://www.nrbook.com/a/bookcpdf.php", notes={"Function gser()"})
    protected static double incompleteGammaSeriesExpansion(double a, double x) {
        int MAX_ITERATIONS = 1000;
        double EPS = 3.0E-7;
        double gamser = 0.0;
        if (x <= 0.0) {
            if (x < 0.0) {
                throw new IllegalArgumentException("x must be >= 0.0");
            }
            gamser = 0.0;
        } else {
            int n;
            double sum;
            double gln = MathUtil.logGammaFunction(a);
            double ap = a;
            double del = sum = 1.0 / a;
            for (n = 1; n <= 1000; ++n) {
                sum += (del *= x / (ap += 1.0));
                if (!(Math.abs(del) < Math.abs(sum) * 3.0E-7)) continue;
                gamser = sum * Math.exp(-x + a * Math.log(x) - gln);
                break;
            }
            if (n > 1000) {
                throw new OperationNotConvergedException("a too large, MAX_ITERATIONS too small in seriesExpansion()");
            }
        }
        return gamser;
    }

    @PublicationReference(author={"William H. Press", "Saul A. Teukolsky", "William T. Vetterling", "Brian P. Flannery"}, title="Numerical Recipes in C, Second Edition", type=PublicationType.Book, year=1992, pages={216, 219}, url="http://www.nrbook.com/a/bookcpdf.php")
    public static double incompleteGammaContinuedFraction(double a, double x) {
        LentzMethod lentz = new LentzMethod();
        lentz.initializeAlgorithm(0.0);
        lentz.iterate(1.0, x + 1.0 - a);
        while (lentz.getKeepGoing()) {
            int i = lentz.getIteration();
            double aterm = (double)(-i) * ((double)i - a);
            double bterm = x + (double)(2 * i + 1) - a;
            lentz.iterate(aterm, bterm);
        }
        double gln = MathUtil.logGammaFunction(a);
        double gcf = lentz.getResult();
        double gamma = 1.0 - Math.exp(-x + a * Math.log(x) - gln) * gcf;
        return gamma;
    }

    @PublicationReference(author={"Wikipedia"}, title="Binomial coefficient", type=PublicationType.WebPage, year=2010, url="http://en.wikipedia.org/wiki/Binomial_coefficient")
    public static int binomialCoefficient(int N, int k) {
        return (int)Math.round(Math.exp(MathUtil.logBinomialCoefficient(N, k)));
    }

    public static double logBinomialCoefficient(int N, int k) {
        return MathUtil.logFactorial(N) - MathUtil.logFactorial(k) - MathUtil.logFactorial(N - k);
    }

    public static double logFactorial(int n) {
        if (n < 0) {
            throw new IllegalArgumentException("Factorial must be >= 0");
        }
        if (n <= 1) {
            return 0.0;
        }
        return MathUtil.logGammaFunction((double)n + 1.0);
    }

    @PublicationReference(author={"Wikipedia"}, title="Beta function", type=PublicationType.WebPage, year=2010, url="http://en.wikipedia.org/wiki/Beta_function")
    public static double logBetaFunction(double a, double b) {
        double ga = MathUtil.logGammaFunction(a);
        double gb = MathUtil.logGammaFunction(b);
        double gab = MathUtil.logGammaFunction(a + b);
        return ga + gb - gab;
    }

    @PublicationReferences(references={@PublicationReference(author={"Wikipedia"}, title="Beta function, Incomplete Beta function", type=PublicationType.WebPage, year=2010, url="http://en.wikipedia.org/wiki/Beta_function#Incomplete_beta_function"), @PublicationReference(author={"William H. Press", "Saul A. Teukolsky", "William T. Vetterling", "Brian P. Flannery"}, title="Numerical Recipes in C, Second Edition", type=PublicationType.Book, year=1992, pages={227}, notes={"Function betai"}, url="http://www.nrbook.com/a/bookcpdf.php")})
    public static double regularizedIncompleteBetaFunction(double a, double b, double x) {
        if (x < 0.0 || x > 1.0) {
            throw new IllegalArgumentException("0 <= x <= 1");
        }
        double bt = x == 0.0 || x == 1.0 ? 0.0 : Math.exp(a * Math.log(x) + b * Math.log(1.0 - x) - MathUtil.logBetaFunction(a, b));
        if (x < (a + 1.0) / (a + b + 2.0)) {
            return bt * MathUtil.incompleteBetaContinuedFraction(a, b, x) / a;
        }
        return 1.0 - bt * MathUtil.incompleteBetaContinuedFraction(b, a, 1.0 - x) / b;
    }

    @PublicationReference(author={"William H. Press", "Saul A. Teukolsky", "William T. Vetterling", "Brian P. Flannery"}, title="Numerical Recipes in C, Second Edition", type=PublicationType.Book, year=1992, pages={227}, notes={"Incomplete Beta Function continued fraction terms for Lentz's method"}, url="http://www.nrbook.com/a/bookcpdf.php")
    protected static double incompleteBetaContinuedFraction(double a, double b, double x) {
        double apb = a + b;
        LentzMethod lentz = new LentzMethod();
        lentz.initializeAlgorithm(0.0);
        lentz.iterate(1.0, 1.0);
        while (lentz.getKeepGoing()) {
            double aterm;
            double den;
            double num;
            int m = lentz.getIteration() / 2;
            double ap2m = a + (double)(2 * m);
            if (lentz.getIteration() % 2 != 0) {
                num = -(a + (double)m) * (apb + (double)m) * x;
                den = ap2m * (ap2m + 1.0);
                aterm = num / den;
            } else {
                num = (double)m * (b - (double)m) * x;
                den = (ap2m - 1.0) * ap2m;
                aterm = num / den;
            }
            lentz.iterate(aterm, 1.0);
        }
        if (lentz.isResultValid()) {
            return lentz.getResult();
        }
        System.out.printf("a = %f, b = %f, x = %f\n", a, b, x);
        throw new OperationNotConvergedException("Lentz's Method failed in Beta continuous fraction!");
    }

    @PublicationReference(author={"Wikipedia"}, title="Dirichlet distribution", type=PublicationType.WebPage, year=2009, url="http://en.wikipedia.org/wiki/Dirichlet_distribution", notes={"Multinomial Beta Function found in the \"Probability density function\" section."})
    public static double logMultinomialBetaFunction(Vector input) {
        double logsum = 0.0;
        double inputSum = 0.0;
        for (int i = 0; i < input.getDimensionality(); ++i) {
            double ai = input.getElement(i);
            inputSum += ai;
            logsum += MathUtil.logGammaFunction(ai);
        }
        return logsum -= MathUtil.logGammaFunction(inputSum);
    }
}

