/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.net.URL;
import java.util.ArrayList;

import javax.swing.*;
import javax.swing.filechooser.FileNameExtensionFilter;


public class MainInterface extends JFrame {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	public static int SCALE = 6;
	public static int UNDO_LEVELS = 16;
	public static int DEFAULT_DIM = 15;
	public static String PROGRAM_NAME = "Jowcha";
	
	Grid grid; GridPanel gridgui;
	ClueList clueList; ClueView clueView;
	InfoPanel puzzleInfoPanel;
	//JList<String> poss; // possible words for current slot
	ArrayList<FillOption> options;
	FillView fillView;
	Dictionary dict;
	Font guiFont;
	JLabel statistics;
	JLabel fillIndicator;
	JFileChooser fileChooser;
	FileNameExtensionFilter ipuzFilter, pngFilter, txtFilter;
	File currentFile;
	GridFiller backgroundFillChecker;
	OptionsChecker backgroundOptionsChecker;
	ImageIcon redLED, yellowLED, greenLED;
	private GridHistory gridHistory;
	
	public MainInterface() {
		updateTitle();
		//setSize(720, 480);
		setLocation(100, 100);
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		guiFont = new Font("Arial", Font.PLAIN, 3*SCALE);
		
		dict = new Dictionary("standard");
		dict.addWordList("scored_wordlist.txt");//"src/wlist_match5.txt");
		
		grid = new Grid(DEFAULT_DIM);
		gridgui = new GridPanel(grid, this);
		clueList = new ClueList(grid);
		clueView = new ClueView(clueList);
		puzzleInfoPanel = new InfoPanel();
		fileChooser = new JFileChooser();
		ipuzFilter = new FileNameExtensionFilter("ipuz", "ipuz");
		pngFilter = new FileNameExtensionFilter("png", "png");
		txtFilter = new FileNameExtensionFilter("txt", "txt");
		fileChooser.setFileFilter(ipuzFilter);
		currentFile = null;
		gridHistory = new GridHistory(grid);
		
		options = new ArrayList<FillOption>();
		fillView = new FillView(options);
		fillView.setFont(guiFont);
		
		JTabbedPane rightPane = new JTabbedPane();
		rightPane.addTab("Fill", fillView);
		rightPane.addTab("Clues", clueView);
		//rightPane.addTab("Puzzle info", puzzleInfoPanel);
		
		JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
				gridgui, rightPane);
		add(split, BorderLayout.CENTER);
		
		int ctrlKey = Toolkit.getDefaultToolkit().getMenuShortcutKeyMaskEx();
		KeyStroke ctrlZ = KeyStroke.getKeyStroke(KeyEvent.VK_Z, ctrlKey);
		KeyStroke ctrlY = KeyStroke.getKeyStroke(KeyEvent.VK_Y, ctrlKey);
		KeyStroke ctrlN = KeyStroke.getKeyStroke(KeyEvent.VK_N, ctrlKey);
		KeyStroke ctrlS = KeyStroke.getKeyStroke(KeyEvent.VK_S, ctrlKey);
		KeyStroke ctrlO = KeyStroke.getKeyStroke(KeyEvent.VK_O, ctrlKey);
		KeyStroke ctrlF = KeyStroke.getKeyStroke(KeyEvent.VK_F, ctrlKey); // autofill
		KeyStroke ctrlU = KeyStroke.getKeyStroke(KeyEvent.VK_U, ctrlKey); // suggest
		KeyStroke ctrlR = KeyStroke.getKeyStroke(KeyEvent.VK_R, ctrlKey); // resize grid
		KeyStroke backtick = KeyStroke.getKeyStroke('`'); // toggle black/white
		//KeyStroke period = KeyStroke.getKeyStroke('.'); // alternative toggle black/white
		
		JMenuBar menubar = new JMenuBar();
		JMenu filemenu = new JMenu("File"); menubar.add(filemenu); filemenu.setMnemonic('f');
		JMenu editmenu = new JMenu("Edit"); menubar.add(editmenu); editmenu.setMnemonic('e');
		JMenu fillmenu = new JMenu("Fill"); menubar.add(fillmenu); fillmenu.setMnemonic('i');
		JMenu helpmenu = new JMenu("Help"); menubar.add(helpmenu); helpmenu.setMnemonic('h');
		
		JMenuItem newitem = new JMenuItem("New"); filemenu.add(newitem);
		newitem.setAccelerator(ctrlN);
		JMenuItem openitem = new JMenuItem("Open"); filemenu.add(openitem);
		openitem.setAccelerator(ctrlO);
		JMenuItem saveitem = new JMenuItem("Save"); filemenu.add(saveitem);
		saveitem.setAccelerator(ctrlS);
		JMenuItem saveasitem = new JMenuItem("Save As..."); filemenu.add(saveasitem);
		JMenu exportItem = new JMenu("Export"); filemenu.add(exportItem);
		JMenuItem exportClues = new JMenuItem("Clues and answers (.txt)"); exportItem.add(exportClues);
		JMenuItem exportCluesOnly = new JMenuItem("Clues only (.txt)"); exportItem.add(exportCluesOnly);
		JMenuItem exportFilledGrid = new JMenuItem("Filled grid (.png)"); exportItem.add(exportFilledGrid);
		JMenuItem exportEmptyGrid = new JMenuItem("Empty grid (.png)"); exportItem.add(exportEmptyGrid);
		
		JMenuItem undoItem = new JMenuItem("Undo");
		undoItem.setAccelerator(ctrlZ);
		JMenuItem redoItem = new JMenuItem("Redo");
		redoItem.setAccelerator(ctrlY);
		JMenuItem resizeGridItem = new JMenuItem("Resize grid");
		resizeGridItem.setAccelerator(ctrlR);
		JMenuItem toggleBlackItem = new JMenuItem("Toggle black/white (backquote)");
		toggleBlackItem.setAccelerator(backtick);
		//toggleBlackItem.setAccelerator(period);
		
		JMenuItem nextSlotItem = new JMenuItem("Suggest next slot");
		nextSlotItem.setAccelerator(ctrlU);
		JMenuItem autoFillItem = new JMenuItem("Auto Fill");
		autoFillItem.setAccelerator(ctrlF);
		
		editmenu.add(undoItem);
		undoItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (gridHistory.canUndo()) {
					grid = gridHistory.undo();
					gridgui.grid= grid;
					gridgui.repaint();
					updateStatistics();
					updateClues();
					checkFill();
				}
			}
		});
		editmenu.add(redoItem);
		redoItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (gridHistory.canRedo()) {
					grid = gridHistory.redo();
					gridgui.grid = grid;
					gridgui.repaint();
					updateStatistics();
					updateClues();
					checkFill();
				}
			}
		});
		editmenu.add(resizeGridItem);
		resizeGridItem.addActionListener(new ActionListener () {
			public void actionPerformed(ActionEvent e) {
				String input = JOptionPane.showInputDialog("New side length (must be odd):");
				Integer newDim = Integer.parseInt(input);
				boolean success = false;
				if (newDim != null) {
					success = grid.setDim(newDim);
					gridgui.setCursor(0,0);
					gridgui.repaint();
					gridHistory.push(grid);
					updateStatistics();
					checkFill();
				}
				if (!success) {
					JOptionPane.showMessageDialog(gridgui, "Invalid dimension");
				}
			}
		});
		editmenu.add(toggleBlackItem);

		ActionListener toggleAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				gridgui.toggle();
			}
		};
		toggleBlackItem.addActionListener(toggleAction);
		
		
		fillmenu.add(nextSlotItem);
		nextSlotItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				Coords next = grid.getNextToFill(dict);
				gridgui.setActiveWord(next);
			}
		});
		fillmenu.add(autoFillItem);
		autoFillItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				Grid result = grid.autoFill(dict);
				if (result == null) {
					//System.out.println("No fill found.");
				}
				else {
					grid = result;
					gridgui.grid = grid;
					updateClues();
					gridHistory.push(grid);
					gridgui.repaint();
				}
			}
		});
		
		abstract class FileListener implements ActionListener {
			FileNameExtensionFilter filter;
			public FileListener(FileNameExtensionFilter filter) {
				this.filter = filter;
			}
			public File getSaveFile() {
				/* Return file with extension enforced, or null if canceled. */
				fileChooser.setFileFilter(filter);
				if (fileChooser.showSaveDialog(menubar) == JFileChooser.APPROVE_OPTION) {
					File selectedFile = fileChooser.getSelectedFile();
					if (fileChooser.getFileFilter() == filter &&
							!filter.accept(selectedFile)) {
						String newName = selectedFile.getAbsolutePath() + "." + filter.getExtensions()[0];
						selectedFile = new File(newName);
					}
					return selectedFile;
				}
				return null;
			}
			public File getOpenFile() {
				/* Get selected file to open, or null if none selected */
				fileChooser.setFileFilter(filter);
				if (fileChooser.showOpenDialog(menubar) == JFileChooser.APPROVE_OPTION) {
					return fileChooser.getSelectedFile();
				}
				return null;
			}
		}
			
		class IPuzFileListener extends FileListener {
			boolean save; boolean saveas;
			IPuzFileListener(boolean save, boolean saveas) {
				super(ipuzFilter);
				this.save=save; this.saveas=saveas;
			}
			public void actionPerformed(ActionEvent e) {
				try {
					if (saveas || (save && currentFile == null)) { // Save as
						File selectedFile = getSaveFile();
						if (selectedFile != null) {
							CrosswordFile xf = new CrosswordFile(grid, clueList);
							xf.saveIpuz(selectedFile);
							currentFile = selectedFile;
							updateTitle();
						}
					}
					
					else if (save && currentFile != null) { // Save
						CrosswordFile xf = new CrosswordFile(grid, clueList);
						xf.saveIpuz(currentFile);
						updateTitle();
					}
					else { // Open
						File selectedFile = getOpenFile();
						if (selectedFile != null) {
							currentFile = selectedFile;
							CrosswordFile xf = new CrosswordFile(currentFile);
							grid = xf.getGrid();
							clueList.setAll(xf.getClueList());
							gridgui.setGrid(grid);
							updateTitle();
							gridgui.repaint();
							updateClues();
							updateStatistics();
							checkFill();
						}
					}
				} catch (Exception ex) {}
			}
		}
		class PngFileListener extends FileListener {
			private boolean withFill;
			public PngFileListener(boolean withFill) {
				super(pngFilter);
				this.withFill = withFill;
			}

			@Override
			public void actionPerformed(ActionEvent arg0) {
				File f = getSaveFile();
				try {
				if (f != null) {
					gridgui.saveImage(f,  withFill);
				}} catch(Exception ex) {}
			}
			
		}
		class TxtFileListener extends FileListener {
			boolean withAnswers;
			public TxtFileListener(boolean withAnswers) {
				super(txtFilter);
				this.withAnswers= withAnswers;
			}

			@Override
			public void actionPerformed(ActionEvent arg0) {
				File f = getSaveFile();
				try {
					if (f != null) {
						clueList.saveClues(f, withAnswers);
					}
				} catch (Exception e) {}
				
			}
		}
		
		openitem.addActionListener(new IPuzFileListener(false,false));
		saveitem.addActionListener(new IPuzFileListener(true,false));
		saveasitem.addActionListener(new IPuzFileListener(true,true));
		newitem.addActionListener(new ActionListener () {
			public void actionPerformed(ActionEvent arg0) {
				grid = new Grid(DEFAULT_DIM);
				gridgui.setGrid(grid);
				gridgui.repaint();
				updateStatistics();
				updateClues();
				clueList.clearHints();
				checkFill();
				gridHistory.push(grid);
				currentFile = null;
				updateTitle();
			}
		});
		exportFilledGrid.addActionListener(new PngFileListener(true));
		exportEmptyGrid.addActionListener(new PngFileListener(false));
		exportClues.addActionListener(new TxtFileListener(true));
		exportCluesOnly.addActionListener(new TxtFileListener(false));
		
		setJMenuBar(menubar);
		
		statistics = new JLabel("");
		redLED = makeIcon("redLED.png", "Red LED");
		yellowLED = makeIcon("yellowLED.png", "Yellow LED");
		greenLED = makeIcon("greenLED.png", "Green LED");
		fillIndicator = new JLabel("", yellowLED, JLabel.LEFT);
		updateStatistics();
		checkFill();
		statistics.setFont(guiFont);
		fillIndicator.setFont(guiFont);
		JPanel bottomPanel = new JPanel(new BorderLayout());
		bottomPanel.add(statistics, BorderLayout.LINE_END);
		bottomPanel.add(fillIndicator, BorderLayout.LINE_START);
		this.add(bottomPanel, BorderLayout.PAGE_END);
		
		pack();
		setVisible(true);
	}
	
	private void updateTitle() {
		String filename = "Untitled";
		if (currentFile != null) {filename = currentFile.getName();}
		setTitle(String.format("%s - %s", filename, PROGRAM_NAME));		
	}

	private class GridHistory {
		Grid[] history;
		int oldestIndex, currentIndex;
		
		public GridHistory(Grid initialGrid) {
			history = new Grid[UNDO_LEVELS];
			history[0] = new Grid(initialGrid);
			oldestIndex = 0; currentIndex = 0;
		}
		public void push(Grid g) {
			++currentIndex;
			history[currentIndex % UNDO_LEVELS] = new Grid(g);
			if (currentIndex%UNDO_LEVELS == oldestIndex%UNDO_LEVELS) {
				oldestIndex = (oldestIndex+1)%UNDO_LEVELS;
			}
			for (int k = currentIndex+1; k%UNDO_LEVELS != oldestIndex%UNDO_LEVELS; ++k) {
				history[k % UNDO_LEVELS] = null;
			}
			//System.out.println(currentIndex);
			//System.out.println(oldestIndex);
		}
		public boolean canUndo() {
			return currentIndex%UNDO_LEVELS != oldestIndex%UNDO_LEVELS;
		}
		public boolean canRedo() {
			return ((currentIndex+1)%UNDO_LEVELS != oldestIndex%UNDO_LEVELS) &&
					history[(currentIndex+1)%UNDO_LEVELS] != null;
		}
		public Grid undo() {
			if (!canUndo()) return null; // reached undo limit
			currentIndex = (currentIndex-1+UNDO_LEVELS)%UNDO_LEVELS; // ensure > 0
			//System.out.println(currentIndex);
			return new Grid(history[currentIndex]);
		}
		public Grid redo() {
			if (!canRedo()) return null;
			return new Grid(history[(++currentIndex) % UNDO_LEVELS]);
		}
	}
	public void saveState() { // for undo/redo
		gridHistory.push(grid);
	}
	
	private ImageIcon makeIcon(String filename, String description) {
		URL url = getClass().getResource(filename);
		if (url == null) return null;
		return new ImageIcon(url, description);
	}
	
	public void updatePoss(String query) {
		options = dict.lookup(query);
		// We want a new object because a background thread may be looping over
		// the existing one.
		fillView.setFillOptions(options);
		fillView.revalidate();
		fillView.repaint();
		fillView.repaintTable();
		checkOptions();
	}
	
	public void updateStatistics() {
		/* Display the number of words and blocks. */
		int numBlocks = grid.countBlocks();
		int numWords = grid.countWords();
		statistics.setText(String.format("<html>Words: %d/78<br/>Blocks: %d/42</html>",
				numWords, numBlocks));
	}
	
	public void updateFillIndicator(int possibility) {
		/* Update the indicator depending on whether there was no fill found,
		 * was a fill found, or if we are currently searching in another thread.
		 * GridFiller.FOUND, NOT_FOUND, SEARCHING
		 */
		String text = "Fill not found";
		ImageIcon icon = redLED;
		if (possibility == GridFiller.FOUND) {text = "Fill possible"; icon=greenLED;}
		else if (possibility == GridFiller.SEARCHING) {text = "Searching for fill..."; icon=yellowLED;}
		fillIndicator.setText(text);
		fillIndicator.setIcon(icon);
	}
	
	//public void updatePoss(ArrayList<String> words) {
		//poss.setListData(words.toArray(new String[words.size()]));
	//}
	
	public void checkFill() {
		/* Update fill indicator. */
		// Start a thread in the background to determine whether autofill
		// would find a solution.
		updateFillIndicator(GridFiller.SEARCHING);
		if (backgroundFillChecker != null) {
			backgroundFillChecker.stopFlag.stopAutoFill = true;
		}
		backgroundFillChecker = new GridFiller(grid, this, dict);
		Thread bgThread = new Thread(backgroundFillChecker);
		bgThread.start();
	}
	public void checkOptions() {
		/* Loop over fill options in fill tab and check them in the background. */
		if (backgroundOptionsChecker != null) {
			backgroundOptionsChecker.stopFlag.stopAutoFill = true;
		}
		Coords slot = gridgui.getSelectedSlot();
		backgroundOptionsChecker = new OptionsChecker(grid, slot, options, dict, fillView);
		Thread bgOptThread = new Thread(backgroundOptionsChecker);
		bgOptThread.start();
	}
	public void updateClues() {
		clueList.update(grid);
		clueView.revalidate(); // reallocates rectangle to draw on
		clueView.repaint(); // redraws
	}
	
	public static void main(String[] args) {
		new MainInterface();		
	}
}
