/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

import javax.imageio.ImageIO;
import javax.swing.*;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

public class GridPanel extends JPanel {
	/**
	 * Visual representation of a grid and user navigation on it.
	 */
	private static final long serialVersionUID = 1L;
	MainInterface gui;
	Grid grid;
	
	static Font capsFont = new Font("Arial", Font.PLAIN, 4*MainInterface.SCALE);
	static Font clueNumFont = new Font("Arial", Font.PLAIN, 2*MainInterface.SCALE);
	static Color cursorLight = new Color(255, 255, 200);
	static Color cursorMed = new Color(255, 200, 120);
	static Color cursorDark = new Color(112, 91, 44);
	
	int sqsize = 6; // in MainInterface.SCALE units
	static int XPAD = MainInterface.SCALE*2;
	static int YPAD = -MainInterface.SCALE;
	String message = "";

	int row = 0; int col = 0; // indices of "active" square
	boolean horiz = true; // are we going horizontally?
	
	public GridPanel(Grid grid, MainInterface gui) {
		this.grid = grid;
		this.gui = gui;
		
		addKeyListener(new GridKeys(this));
		addMouseListener(new GridMouse(this));
	}
	
	public void setGrid(Grid g) {grid = g;}
	
	public void setCursor(int r, int c) {row=r;col=c;}
	
	public void enterChar(char ch) {
	/* Try to update square character. Grid checks whether we can do this.
	 * If so, update current position & repaint. */
		if (grid.enterChar(row, col, ch)) {
			whiteMove(horiz ? 1 : 0, horiz ? 0 : 1); // calls repaint
			gui.checkFill();
			gui.updateClues();
			gui.saveState();
		}
	}
	
	public void setActiveWord(Coords c) {
		row = c.r; col = c.c; horiz = c.dirhoriz;
		repaint();
	}
	public Coords getSelectedSlot() {
		return new Coords(row,col,horiz);
	}
	
	public void backspace() {
		if (grid.enterChar(row, col, ' ')) {
			whiteMove(horiz ? -1 : 0, horiz ? 0 : -1);
			gui.checkFill();
			gui.updateClues();
			gui.saveState();
		}
	}
	
	public void delete() {
		if (grid.enterChar(row, col, ' ')) {
			whiteMove(horiz ? 1 : 0, horiz ? 0 : 1);
			gui.checkFill();
			gui.updateClues();
			gui.saveState();
		}
	}
	
	// try to move, but if there's a black square there then don't.
	// Don't go off the edge of the board.
	public void whiteMove(int dx, int dy) {
		int newc = col + dx; int newr = row + dy;
		if (!grid.onGrid(newr, newc) || grid.isBlack(newr, newc)) {
			repaint();
			return;
		}
		row = newr; col = newc;
		repaint();		
	}
	
	// try to move. Don't go off the edge of the board.
	// If on a white square and orientation doesn't match,
	// change orientation and don't move
	// Require at most one argument be nonzero for anything to happen.
	public void move(int dx, int dy) {
		// first check orientation match
		if (!grid.isBlack(row, col) && ((dy != 0 && horiz) || (dx != 0 && !horiz))) {
			horiz = !horiz;
			repaint();
			return;
		}
		// do the move itself
		col += dx; row += dy;
		// set word orientation
		horiz = (dy == 0);
		
		// clip to inside grid
		row = grid.clipToDim(row);
		col = grid.clipToDim(col);
		
		repaint();
	}
	
	// toggle current square
	public void toggle() {
		grid.toggle(row, col);
		repaint();
		gui.updateStatistics();
		gui.checkFill();
		gui.updateClues();
		gui.saveState();
	}
	
	public String currentWord() {
		return grid.getWord(row, col, horiz);
	}

	public Dimension getPreferredSize() {
		int dim = grid.getDim();
		return new Dimension(dim*sqsize*MainInterface.SCALE, dim*sqsize*MainInterface.SCALE);
	}
	//public Dimension getMinimumSize() {
	//	return getPreferredSize();
	//}
	
	public int getSquarePx() {return MainInterface.SCALE*sqsize;}
	
	public void saveImage(File f, boolean filled) throws IOException {
		Dimension dim = getPreferredSize();
		BufferedImage im = new BufferedImage(dim.width, dim.height, BufferedImage.TYPE_INT_RGB);
		paintComponent(im.getGraphics(), false, filled, false);
		ImageIO.write(im, "png", f);
	}

	public void paintComponent(Graphics g) {
		paintComponent(g, true, true, true);
	}
	public void paintComponent(Graphics g, boolean withHighlight, boolean withFill, boolean refreshGUI) {
		super.paintComponent(g);
		
		// probably want to refresh list of possibilities
		// HACK: If we need to do this update then we will also need to repaint,
		// but it's not a logical component of the repaint itself.
		if (refreshGUI) {
			gui.updatePoss(grid.getWord(row, col, horiz));
		}
		
		int sqpx = MainInterface.SCALE*sqsize; // square size in pixels
		
		// draw a white background
		g.setColor(Color.WHITE);
		int dim = grid.getDim();
		g.fillRect(0, 0, sqsize*dim*MainInterface.SCALE, sqsize*dim*MainInterface.SCALE);
		
		// fill in the current word square highlights
		if (withHighlight) {
			g.setColor(cursorLight);
			ArrayList<Coords> wordsquares = grid.getWordSquares(row, col, horiz);
			for (Coords c : wordsquares) {
				g.fillRect(c.c*sqpx, c.r*sqpx, sqpx, sqpx);
			}
			
			// fill in the active square highlight
			g.setColor(grid.isBlack(row, col) ? cursorDark : cursorMed);
			g.fillRect(col*sqpx, row*sqpx, sqpx, sqpx);
		}
		
		// now do all the squares: border, black fill, and letters
		g.setColor(Color.BLACK);
		
		for (int j=0; j < dim; ++j) { // row
		for (int k = 0; k < dim; ++k) { // col
			// draw a black border
			g.drawRect(k*sqpx, j*sqpx, sqpx, sqpx);
			boolean isBlack = grid.isBlack(j, k);
			if (!isBlack) {
				if (withFill) {
					g.setFont(capsFont);
					g.drawString(
						String.valueOf(Character.toUpperCase(grid.getChar(j, k))),
						k*sqpx+XPAD, (j+1)*sqpx+YPAD	
					);
				}
				g.setFont(clueNumFont);
				Integer clueNum = grid.clueNumbers[j][k];
				if (clueNum != null) g.drawString(grid.clueNumbers[j][k].toString(),
						k*sqpx, j*sqpx + clueNumFont.getSize());
			}
			// Fill in black if called for. Don't fill in black
			// if we're on the active square, unless withHighlight is false.
			if (isBlack && (!(j == row && k == col) || !withHighlight)) {
				g.fillRect(k*sqpx, j*sqpx, sqpx, sqpx);
			}
		}
		}
	}
}