/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;

class Node {
	public Node parent;
	public char ch;
	public int depth; // root is at depth -1. So the depth is the character index in word
	public boolean isWord;
	private int score; // score of the word at this node, if there is one.
	public Hashtable<Character,Node> edges;
	
	public Node(Node par, char c, int d, boolean isw, int score) {
		parent = par;
		ch = Character.toLowerCase(c);
		depth = d;
		isWord = isw;
		this.score = score;
		edges = new Hashtable<Character,Node>();
	}
	
	String getWord() {
		StringBuilder sb = new StringBuilder();
		
		Node i = this;
		while (i.parent != null) {
			sb.insert(0, i.ch);
			i = i.parent;
		}
		
		return sb.toString();
	}
	int getScore() {
		return score;
	}
	FillOption getFillOption() {
		return new FillOption(getWord(), getScore());
	}
}

public class Dictionary {
	public static int DEFAULT_SCORE = 1;
	public static String SCORE_DELIMITER = ";";
	
	Node root;
	String description;
	HashMap<String, Integer> countWordsMemoTable;
	
	public Dictionary(String des) {
		root = new Node(null, '\0', -1, false, 0);
		countWordsMemoTable = new HashMap<String,Integer>();
	}
	
	public void addWordList(String filename) {
		try {
			InputStream s= getClass().getResourceAsStream(filename);
			BufferedReader br = new BufferedReader(new InputStreamReader(s));
			String nextline = br.readLine(); // not including \n
			while (nextline != null) {
				if (nextline.charAt(0) == '#') {nextline= br.readLine(); continue;} // comment in wordlist
				String[] wordandscore = nextline.split(SCORE_DELIMITER);
				String nextword = wordandscore[0]; int score = DEFAULT_SCORE;
				if (wordandscore.length > 1) {
					score = Integer.parseInt(wordandscore[1]);
				}
			
				addWord(nextword, score);
				nextline = br.readLine();
			}
			br.close();
		}
		catch (FileNotFoundException e) { System.out.println("No such file."); }
		catch (IOException e) { System.out.println("File error."); }
		
	}
	
	// May be called multiple times on the same word, with no adverse effects.
	public void addWord(String word, int score) {
		word = word.toLowerCase();
		Node curr = root;
		Node next;
		char ch; // next character
		for (int i = 0; i < word.length(); ++i) {
			ch = word.charAt(i);
			next = curr.edges.get(ch);
			if (next == null) { // no edge for that character
				// make one
				next = new Node(curr, ch, i, false, score);
				curr.edges.put(ch, next);
			}
			curr = next;
		}
		curr.isWord = true; // mark it as a word
	}
	
	public int getWordScore(String query) {
		/* Return the score of the queried word, or DEFAULT_SCORE if not recognized. */
		ArrayList<FillOption> res = lookup(query);
		if (res.size() == 0) return DEFAULT_SCORE;
		return res.get(0).getWordScore();
	}
	
	public ArrayList<FillOption> lookup(String query) {
		/* Return candidate fills, sorted by score. Ignore case. Spaces are wildcards. */
		ArrayList<FillOption> matches = new ArrayList<FillOption>();
		if (query == "") return matches;
		lookup(query.toLowerCase(), root, matches);
		matches.sort(FillOptionCompare.getInstance());
		return matches;
	}
	
	public int countWords(String query) {
		/* Count the number of options for the query string.
		 * Spaces are wildcards. */
		Integer memoLookup = countWordsMemoTable.get(query.toLowerCase());
		if (memoLookup == null) {
			int size = lookup(query).size();
			countWordsMemoTable.put(query, size);
			return size;
		}
		return memoLookup;
	}
	
	public void lookup(String query, Node n, ArrayList<FillOption> matches) {
	/* query is a combination of lowercase characters and wildcards (spaces).
	 * Start at node n and put all matching results into matches.
	 * Precondition: the n.depth character of want matches n.ch. */
		// do a DFS search
		if (n.depth == query.length()-1) {
			// We've come to the end of our search query.
			if (n.isWord) {
				matches.add(0, n.getFillOption());
			}
			return; // return either way
		};
		
		char nextChar = query.charAt(n.depth+1);
		if (nextChar == ' ') { // Wildcard. Search all child nodes
			for (Node k : n.edges.values()) {
				lookup(query, k, matches);
			}
		}
		else {
			Node k = n.edges.get(nextChar);
			if (k != null) {
				lookup(query, k, matches);
			}
		}
	}
}
