/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;

class Clue {
	public Coords coords; // row, column, direction (across = true).
	public int clueNumber;
	public String answer;
	public String hint;
	
	public Clue(int row, int col, boolean isAcross, int clueNumber, String answer, String hint) {
		this(new Coords(row, col, isAcross), clueNumber, answer, hint);
	}
	public Clue(Coords coords, int clueNumber, String answer, String hint) {
		this.coords=coords; this.clueNumber=clueNumber; this.answer=answer; this.hint=hint;
	}
}

public class ClueList {
	/* Maintains clue labels and clues for a particular Grid. */
	private Grid grid;
	private ArrayList<Clue> clues;
	private HashMap<Coords,Clue> coordsToClue;
	
	public ClueList(Grid g) {
		grid = g;
		clues = new ArrayList<Clue>();
		coordsToClue = new HashMap<Coords,Clue>();
		
		for (int row=0; row<g.dim; ++row) { for (int col=0; col<g.dim; ++col) {
			Coords coords; Clue thisClue;
			if (grid.hasAcrossClue(row,col)) {
				coords = new Coords(row,col,true);
				thisClue = new Clue(coords,
						g.clueNumbers[row][col],grid.getWord(row,col,true),"");
				clues.add(thisClue);
				coordsToClue.put(coords, thisClue);
			}
			if (grid.hasDownClue(row, col)) {
				coords = new Coords(row,col,false);
				thisClue = new Clue(coords,
						g.clueNumbers[row][col],grid.getWord(row,col,false),"");
				clues.add(thisClue);
				coordsToClue.put(coords, thisClue);
			}
		}}
	}
	public void setHint(int clueNumber, boolean isAcross, String hint) {
		// look for an existing clue matching clueNumber and isAcross.
		// if there is one, replace the hint with the provided one.
		for (Clue clue : clues) {
			if (clueNumber == clue.clueNumber && clue.coords.dirhoriz == isAcross) {
				clue.hint = hint;
				//System.out.println(hint);
				return;
			}
		}
	}
	public void setAll(ClueList other) {
		/* Copy all clues from other into this. */
		grid=other.grid;
		clues = other.clues;
		coordsToClue = other.coordsToClue;
	}
	public int size() {
		return clues.size();
	}
	public int getNumber(int k) {return clues.get(k).clueNumber;}
	public String getDirection(int k) {return clues.get(k).coords.dirhoriz ? "Across" : "Down";}
	public boolean isAcross(int k) {return clues.get(k).coords.dirhoriz;}
	public String getAnswer(int k) {return clues.get(k).answer;}
	public String getHint(int k) {return clues.get(k).hint;}
	public void setHint(int k, String newHint) {
		Clue clue = clues.get(k);
		if (clue != null) clue.hint = newHint;
	}
	
	private void merge(ClueList newClueList) {
		/* Use the numbering and answers from the new list, but substitute in
		 * hints from this where possible. newClueList will contain the merged result. */
		for (Clue clue : newClueList.clues) {
			boolean inOldList = coordsToClue.containsKey(clue.coords);
			if (inOldList) {
				String oldHint = coordsToClue.get(clue.coords).hint;
				clue.hint = oldHint;
			}
		}
	}
	public void update(Grid g) {
		/* Update the ClueList to reflect the possibly changed Grid. */
		ClueList updatedList = new ClueList(g);
		merge(updatedList);
		grid = g;
		clues = updatedList.clues;
		coordsToClue = updatedList.coordsToClue;
	}
	public void clearHints() {
		/* Remove all hints from the clues */
		for (Clue c : clues) {c.hint = "";}
	}
	
	public ArrayList<Clue> getCluesByDirection(boolean across) {
		ArrayList<Clue> result = new ArrayList<Clue>();
		for (Clue c : clues) {
			if (c.coords.dirhoriz == across) result.add(c);
		}
		return result;
	}
	public void saveClues(File f, boolean withAnswers) throws IOException {
		int CLUE_WIDTH = 80; // in characters
		FileWriter writer = new FileWriter(f);
		boolean dir = true;
		do { // loop over dir = true, then false
			String type= "ACROSS";
			if (!dir) type= "DOWN";
			writer.write(type + "\n");
			for (Clue c : getCluesByDirection(dir)) {
				if (withAnswers) {
					writer.write(String.format(
						"%-3d%-"+Integer.toString(CLUE_WIDTH)+"s\t%s\n",
						c.clueNumber,c.hint,c.answer.toUpperCase()));
				} else {
					writer.write(String.format("%-3d%s\n", c.clueNumber,c.hint));
				}
			}
			if (dir) {writer.write("\n");}
			dir = !dir;
		} while (!dir);
		writer.close();
		
	}
}
