/* Copyright © 2021 Salvatore S. Elder Jr.
 * Under an MIT license
 * See LICENSE file for details
 */

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.PriorityQueue;

class WordScore {
	public String word;
	public double score;
	public WordScore(String w, double s) {word = w; score = s;}
}

class WordCompare implements Comparator<WordScore> {
	public int compare(WordScore a, WordScore b) {
		if (b.score>a.score) return 1;
		if (b.score<a.score) return -1;
		return 0;
	}
}

class StopFlag {
	public volatile boolean stopAutoFill = false;
}

class Coords {
	public int r,c;
	public boolean dirhoriz;
	public Coords(int r, int c) {this.r=r; this.c=c;}
	public Coords(int r, int c, boolean dir) {this.r=r;this.c=c;this.dirhoriz=dir;}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == this) return true;
		if (obj == null || obj.getClass() != this.getClass()) return false;
		return (r==((Coords)obj).r && c==((Coords)obj).c && dirhoriz==((Coords)obj).dirhoriz);
	}
	@Override
	public int hashCode() {
		return Integer.valueOf(r).hashCode()
				^ (Integer.valueOf(c).hashCode() << 4)
				^ (Boolean.valueOf(dirhoriz).hashCode() << 8);
	}
}

public class Grid {
	/* Class invariant:
	 * 	- clueNumbers and clueDirections consistent with black squares. */
	public static int MAX_WORDS = 100;
	public static char BLOCK_CHAR = '#';
	public static char BLANK_CHAR = ' ';
	
	public static int DOWN = 0;
	public static int ACROSS = 1;
	public static int BOTH = 2;
	//public static int MAX_WORDS_FILL = 10;
	
	char[][] squares; // row, col-indexed, with row,col=0 the top-left square.
	Integer[][] clueNumbers;
	ClueDirection[][] clueDirections;
	int dim; // number of squares to a side
	
	/* Some parameters used in backjumping during autofill. */
	private Coords nextToFill = null;
	private PriorityQueue<WordScore> candidatesRanked = null;
	public HashMap<Coords,Grid> backJumpTo;
	
	public enum ClueDirection {NEITHER, BOTH, ACROSS, DOWN};
	
	public Grid(int d) {
		dim = d;
		squares = new char[dim][dim];
		clueNumbers = new Integer[dim][dim];
		clueDirections = new ClueDirection[dim][dim];
		backJumpTo = new HashMap<Coords,Grid>();
		for (int j= 0; j < dim; ++j)
			for (int k=0; k < dim; ++k)
				squares[j][k] = ' '; // put in an empty white square
		//squares[0][0] = '.';
		//squares[dim-1][dim-1] = '.';
		//squares[0][5] = 's';
		//squares[0][6] = 'a';
		numberSquares();
	}
	
	public Grid(Grid g) {
		/* Deep copy the letters in the grid. */
		this(g.dim);
		for (int row=0;row<g.dim;++row) {
			for(int col=0;col<g.dim;++col) {
				squares[row][col]=g.squares[row][col];
			}
		}
		numberSquares();
	}
	
	public void save(File f) throws IOException {
		FileWriter writer = new FileWriter(f);
		for (int row=0; row<dim; ++row) {
			for (int col=0; col<dim; ++col) {
				writer.write(squares[row][col]);
			}
		}
		writer.close();
	}
	public void load(File f) throws IOException {
		FileReader reader = new FileReader(f);
		for (int row=0; row<dim; ++row) {
			for (int col=0; col<dim; ++col) {
				squares[row][col] = (char) reader.read();
			}
		}
		reader.close();
		numberSquares();
	}
	
	public boolean onGrid(int row, int col) {
		return row < dim && col < dim && row >= 0 && col >= 0;
	}
	
	/* Sanitize. Check if black. Update square character.
	 * Returns true iff successful. */
	public boolean enterChar(int row, int col, char ch) {
		ch = Character.toLowerCase(ch);
		if (!(ch == ' ' || Character.isAlphabetic(ch))) return false; // garbage input
		if (isBlack(row,col)) return false; // black square
		squares[row][col] = ch;
		return true;
	}
	
	public char getChar(int row, int col) {
		return squares[row][col];
	}
	public char getChar(Coords c) {return getChar(c.r, c.c);}
	
	public boolean isBlack(int row, int col) {
		return squares[row][col] == Grid.BLOCK_CHAR;
	}
	
	public int countBlocks() {
		int count = 0;
		for (int row=0; row<dim; ++row) {
			for (int col=0; col<dim; ++col) {
				count += (isBlack(row,col) ? 1 : 0);
			}
		}
		return count;
	}
	public int countWords(int index, boolean searchWithinRow) {
		// Count the number of words in the row or column specified.
		boolean prevBlack = true; int count = 0;
		for (int k = 0; k < dim; ++k) {
			boolean thisBlack = isBlack(k,index);
			if (searchWithinRow) thisBlack = isBlack(index, k);
			if (prevBlack && !thisBlack) ++count;
			prevBlack = thisBlack;
		}
		return count;
	}
	public int countWords() {
		// Count the total number of words in the puzzle.
		int count = 0;
		for (int row=0; row<dim; ++row) count += countWords(row,true);
		for (int col=0; col<dim; ++col) count += countWords(col,false);
		return count;
	}
	private void numberSquares() {
		/* Ensure that clueNumbers and clueDirections are correct. */
		int nextNum = 1;
		for (int row=0; row<dim;++row) {
			for (int col=0; col<dim; ++col) {
				// if white and begins a word, number the square.
				boolean white = !isBlack(row,col);
				boolean startsH = white && (col == 0 || isBlack(row,col-1));
				boolean startsV = white && (row==0 || isBlack(row-1,col));
				if (startsH || startsV) {
					clueNumbers[row][col] = nextNum++;
				} else {clueNumbers[row][col]=null;}
				
				ClueDirection cd = ClueDirection.NEITHER;
				if (startsH && startsV) {cd = ClueDirection.BOTH;}
				else if (startsH) {cd = ClueDirection.ACROSS;}
				else if (startsV) {cd = ClueDirection.DOWN;}
				clueDirections[row][col] = cd;
			}
		}
	}
	public boolean hasAcrossClue(int row, int col) {
		/* Return true iff this square has a number associated with an Across clue. */
		ClueDirection cd = clueDirections[row][col];
		return (cd == ClueDirection.ACROSS || cd == ClueDirection.BOTH);
	}
	public boolean hasDownClue(int row,int col) {
		/* Return true iff this square has a number associated with a Down clue. */
		ClueDirection cd = clueDirections[row][col];
		return (cd == ClueDirection.DOWN || cd == ClueDirection.BOTH);
	}
	
	public boolean isEmpty(int row, int col) {
		return squares[row][col] == ' ';
	}
	
	/* If the coordinate is outside the dimension of the grid, clip it onto the grid.
	 * If it's on the grid already, simply return it. */
	public int clipToDim(int coord) {
		if (coord >= dim) return dim - 1;
		if (coord < 0) return 0;
		return coord;
	}
	
	public int getDim() { return dim; }
	
	public boolean setDim(int d) {
		/* Resize the grid. Imagine that we are expanding or shrinking from the edge,
		 * so that the center of the grid remains the same.
		 * Side length must be odd. Return true iff successful. */
		if (d % 2 == 0 || d < 1 || d > 101) return false;
		char[][] newSquares = new char[d][d];
		int pad = (d - dim)/2; // if positive, number of rows to add to the top, say.
		// If there was an old square at (j,k), then the equivalent new square is
		// at (j+pad, k+pad).
		for (int row = 0; row < d; ++row) {
			for (int col = 0; col < d; ++col) {
				int oldRow = row - pad; // index in old squares for equivalent to (row,col) in new squares
				int oldCol = col - pad;
				if (oldRow >= 0 && oldRow < dim && oldCol >= 0 && oldCol < dim) {
					newSquares[row][col] = squares[oldRow][oldCol];
				}
				else {
					newSquares[row][col] = ' ';
				}
			}
		}
		dim = d;
		squares = newSquares;
		clueNumbers = new Integer[dim][dim];
		clueDirections = new ClueDirection[dim][dim];
		numberSquares();
		return true;
	}
	
	// toggle a square's black-whiteness. Enforce symmetry automatically.
	// Check if in range first. Returns true iff successful.
	public boolean toggle(int row, int col) {
		if (!onGrid(row, col)) return false;
		
		char dest = (isBlack(row,col) ? ' ' : Grid.BLOCK_CHAR);
		squares[row][col] = squares[dim-row-1][dim-col-1] = dest;
		numberSquares();
		return true;
	}
	
	public void setChar(int row,int col,char letter) {
		squares[row][col]=letter;
		if (letter == BLOCK_CHAR) {numberSquares();}
	}
	public void setChar(Coords c, char letter) {
		setChar(c.r,c.c,letter);
	}
	
	public boolean wordFits(Coords loc, String word, Dictionary dict) {
		/* Return true iff inserting the provided word into the indicated slot
		 * allows real words to be filled cross-wise. */
		double lookahead = getLookaheadScore(getWordSquares(loc), word, dict);
		return lookahead > 0.1;
	}
	
	public Grid withWord(Coords loc, String word) {
		/* Return a new grid with word filled into the slot which the provided Coords belongs to.
		 * Precondition: word has the appropriate number of characters. */
		Grid result = new Grid(this);
		ArrayList<Coords> squares = getWordSquares(loc);
		for (int k = 0; k < word.length() && k < squares.size(); ++k) {
			result.setChar(squares.get(k), word.charAt(k));
		}
		return result;
	}
	
	public ArrayList<Coords> getWordSquares(int row, int col, boolean dirhoriz) {
		/* Return all squares making up the word which contains row, col and runs
		 * in the direction indicated. If the indicated square is black. */
		ArrayList<Coords> res = new ArrayList<Coords>();
		if (dirhoriz) {
			for (int c = col; c < dim && !isBlack(row,c); ++c) res.add(new Coords(row, c, true));
			for (int c = col-1; c >= 0 && !isBlack(row,c); --c) res.add(0, new Coords(row, c, true));
		}
		else {
			for (int r = row; r < dim && !isBlack(r,col); ++r) res.add(new Coords(r, col, false));
			for (int r = row-1; r >= 0 && !isBlack(r,col); --r) res.add(0, new Coords(r, col, false));
		}
		return res;
	}
	public ArrayList<Coords> getWordSquares(Coords c) {return getWordSquares(c.r,c.c,c.dirhoriz);}
	
	public ArrayList<Coords> getSlots() {
		/* Return the coordinate and direction of the first square
		 * in every word of the puzzle. */
		ArrayList<Coords> res = new ArrayList<Coords>();
		for (int row=0; row < dim; ++row) {
			for (int col=0; col < dim; ++col) {
				if (!isBlack(row, col) && (col == 0 || isBlack(row,col-1)))
					res.add(new Coords(row,col,true));
				if (!isBlack(row, col) && (row == 0 || isBlack(row-1,col)))
					res.add(new Coords(row,col,false));
			}
		}
		return res;
	}
	public ArrayList<String> getWords() {
		/* Return the contents of all slots. */
		ArrayList<String> res = new ArrayList<String>();
		for (Coords c : getSlots()) {
			res.add(getWord(c));
		}
		return res;
	}
	
	public boolean isFilled(Coords firstSquare) {
		/* Determine whether the word starting with c is filled. */
		for (Coords letter : getWordSquares(firstSquare)) {
			if (isEmpty(letter.r, letter.c)) return false;
		}
		return true;
	}
	
	public Coords getNextToFill(Dictionary dict) {
		/* Suggest which slot should be filled next, returning a Coord
		 * containing the location and direction of the first square of that
		 * slot. */
		ArrayList<Coords> slots = getSlots();
		int fewestOptions = Integer.MAX_VALUE;
		Coords result = new Coords(0,0,true);
		for (Coords c : slots) {
			int numOptions = dict.countWords(getWord(c));
			if (numOptions < fewestOptions && !isFilled(c)) {
				fewestOptions = numOptions;
				result = c;
			}
		}
		return result;
	}
	
	public Grid autoFill(Dictionary dict) { return autoFill(null,dict); }
	
	public Grid autoFill(StopFlag flag, Dictionary dict) {
		/* Return a filled Grid if successful, else null if failed.
		 * If a non-null StopFlag is provided, it will be checked at each step
		 * and the search will stop if its stopAutoFill is true.
		 * The plan is:
		 * 1. 	Identify the next slot to fill, based on number of possible fills of that slot
		 *    	alone.
		 *    	List the first MAX_WORDS words that
		 * 		can fit in that slot. If scores are available, pick the MAX_WORDS highest-scored
		 * 		words. For now, just pick the first.
		 * 2. 	Rank those words using look-ahead, i.e., based on how many options will remain for
		 *    	the intersecting slots after choosing a word.
		 * 3. 	Pick the best-scoring one and enter it in `child`. Remove it from `this` so
		 * 		if we jump back we'll try another one.
		 * 		Mark all empty squares in intersecting slots so we know to jump back to `this`
		 * 		if they are later unfillable. Also mark the depth.
		 * 4.	If either the next slot or one of the intersections is unfillable, jump back.
		 * */
		initializeAutoFill(flag,dict);
		Grid result = continueAutoFill(flag,dict);
		if (result != null) {
			result.numberSquares();
		}
		return result;
	}
	
	public boolean isFilled() {
		for (int row=0; row<dim; ++row) {
			for (int col=0; col<dim; ++col) {
				if (isEmpty(row,col)) return false;
			}
		}
		return true;
	}
	
	private void initializeAutoFill(StopFlag flag, Dictionary dict) {
		// Determine the next slot to fill, and rank candidates.
		nextToFill = getNextToFill(dict);
		ArrayList<Coords> squaresList = getWordSquares(nextToFill);
		ArrayList<FillOption> options = dict.lookup(getWord(nextToFill));
		// options already sorted by word score by the Dictionary method.
		// next sort the top MAX_WORDS by lookahead-score.
		candidatesRanked = new PriorityQueue<WordScore>(new WordCompare());
		for (int k=0; k < MAX_WORDS && k < options.size(); ++k) {
			String candidate = options.get(k).getWord();
			double lookaheadScore = getLookaheadScore(squaresList, candidate, dict);
			candidatesRanked.add(new WordScore(candidate, lookaheadScore));
		}
	}
	
	private Grid continueAutoFill(StopFlag flag, Dictionary dict) {
		if (flag != null && flag.stopAutoFill) {
			System.out.println("autofill stopped.");
			return null;
		}
		if (candidatesRanked == null) {
			initializeAutoFill(flag, dict);
		}
		// Check whether the grid has been filled!
		if (isFilled()) return this;
		// at this point it's the same as if we have backjumped here or started here.
		if (candidatesRanked.isEmpty() || candidatesRanked.peek().score < 0.9) { // < 0.9 is safer == 0.
			// no viable path forward. Need to backjump.
			Grid backTarget = backJumpTo.get(nextToFill);
			if (backTarget == null) return null; // search failed!
			return backTarget.continueAutoFill(flag, dict);
		}
		// at this point, we have viable options. Remove the best one from
		// candidatesRanked (so it won't be reconsidered if we back-jump to this
		// node later).
		String chosenWord = candidatesRanked.poll().word;
		Grid childNode = new Grid(dim);
		for (int row=0; row<dim; ++row) {
			for (int col=0; col<dim; ++col) {
				childNode.squares[row][col] = squares[row][col];
			}
		}
		ArrayList<Coords> chosenSquares = getWordSquares(nextToFill);
		for (int k = 0; k < chosenWord.length(); ++k) {
			Coords chosenSquare = chosenSquares.get(k);
			childNode.setChar(chosenSquare, chosenWord.charAt(k));
			for (Coords xSquare : getWordSquares(
					chosenSquare.r, chosenSquare.c, !chosenSquare.dirhoriz)) {
				childNode.backJumpTo.put(xSquare, this);
				for (Coords xxSquare : getWordSquares(xSquare.r,xSquare.c,!xSquare.dirhoriz)) {
					childNode.backJumpTo.put(xxSquare, this);
				}	
			}
		}
		return childNode.autoFill(flag,dict);
	}
	
	public double getLookaheadScore(ArrayList<Coords> squares, String candidate, Dictionary dict) {
		/* Return product of number of options for intersecting words, if we put
		 * candidate into squares. */
		double score = 1.;
		for (int k = 0; k < squares.size(); ++k) {
			Coords square = squares.get(k);
			String query = getWord(square.r, square.c, !square.dirhoriz, candidate.charAt(k));
			if (query.indexOf(BLANK_CHAR) != -1 || getChar(square)==BLANK_CHAR) { // ie, if not already filled.
				score *= dict.countWords(query);
			}
		} return score;
	}
	
	public String getWord(int row, int col, boolean dirhoriz) {
		StringBuilder sb = new StringBuilder();
		ArrayList<Coords> wordsquares = getWordSquares(row, col, dirhoriz);
		for (Coords c : wordsquares) {
			sb.append(squares[c.r][c.c]);
		}
		return sb.toString();
	}
	public String getWord(Coords c) {return getWord(c.r,c.c,c.dirhoriz);}
	
	public String getWord(int row, int col, boolean dirhoriz, char override) {
		/* Get the word, but override the (row, col) square as specified. */
		StringBuilder sb = new StringBuilder();
		ArrayList<Coords> wordsquares = getWordSquares(row, col, dirhoriz);
		for (Coords c : wordsquares) {
			sb.append(c.r == row && c.c == col ? override : squares[c.r][c.c]);
		}
		return sb.toString();
	}
	
	public int getFillScore(Dictionary dict) {
		/* Score a grid based on the average quality of its words. */
		double score = 0.0;
		ArrayList<String> words = getWords();
		for (String w : words) {
			score += dict.getWordScore(w);
		}
		//score = Math.pow(score, 1./words.size());
		return Double.valueOf(score).intValue();
	}
	
	//public ArrayList<String> findWordsFast(int row, int col, boolean dirhoriz, Dictionary dict) {
	//	return dict.lookup(getWord(row, col, dirhoriz));
	//}
	//public ArrayList<String> findWordsFast(Coords c, Dictionary dict) {
	//	return findWordsFast(c.r,c.c,c.dirhoriz,dict);
	//}
}